<?php

use CodeIgniter\Router\RouteCollection;

/**
 * @var RouteCollection $routes
 */

/*
 * --------------------------------------------------------------------
 * API Routes (no authentication required for mobile app sync)
 * --------------------------------------------------------------------
 */
$routes->group('api', ['filter' => 'cors'], function($routes) {
    // API Health Check - returns 200 OK when API is accessible
    $routes->get('', 'Api\AttendanceController::index');

    // Attendance sync endpoints for mobile app
    $routes->post('attendance/sync', 'Api\AttendanceController::sync');
    $routes->post('attendance/sync-batch', 'Api\AttendanceController::syncBatch');
    $routes->post('attendance/validate-records', 'Api\AttendanceController::validateRecords');

    // Sync statistics endpoint
    $routes->get('sync/statistics', 'Api\AttendanceController::statistics');

    // Reports dashboard statistics
    $routes->get('reports/dashboard-stats', 'Reports\ReportController::getDashboardStats');

    // Employee endpoints
    $routes->get('employees', 'Api\EmployeeController::index');
    $routes->get('employees/(:segment)', 'Api\EmployeeController::show/$1');
});

/*
 * --------------------------------------------------------------------
 * Public Routes (Guest)
 * --------------------------------------------------------------------
 */
$routes->group('/', ['filter' => 'guest'], function($routes) {
    $routes->get('', 'Auth\AuthController::login');
    $routes->get('login', 'Auth\AuthController::login');
    $routes->post('login', 'Auth\AuthController::attemptLogin');
    $routes->get('forgot-password', 'Auth\AuthController::forgotPassword');
    $routes->post('forgot-password', 'Auth\AuthController::sendResetLink');
    $routes->get('reset-password/(:any)', 'Auth\AuthController::resetPassword/$1');
    $routes->post('reset-password', 'Auth\AuthController::updatePassword');
});

/*
 * --------------------------------------------------------------------
 * Authenticated Routes
 * --------------------------------------------------------------------
 */
$routes->group('/', ['filter' => 'auth'], function($routes) {
    // Logout
    $routes->get('logout', 'Auth\AuthController::logout');

    // Change Password
    $routes->get('change-password', 'Auth\AuthController::changePassword');
    $routes->post('change-password', 'Auth\AuthController::updateUserPassword');

    // Dashboard
    $routes->get('dashboard', 'Dashboard\DashboardController::index', ['filter' => 'permission:view-dashboard']);

    // Profile
    $routes->get('profile', 'Dashboard\DashboardController::profile');
    $routes->post('profile/update', 'Dashboard\DashboardController::updateProfile');

    /*
     * Employee Management
     */
    $routes->group('employees', ['filter' => 'permission:view-employees'], function($routes) {
        $routes->get('', 'Employees\EmployeeController::index');
        $routes->get('view/(:num)', 'Employees\EmployeeController::view/$1');
        $routes->get('create', 'Employees\EmployeeController::create', ['filter' => 'permission:create-employee']);
        $routes->post('store', 'Employees\EmployeeController::store', ['filter' => 'permission:create-employee']);
        $routes->get('edit/(:num)', 'Employees\EmployeeController::edit/$1', ['filter' => 'permission:edit-employee']);
        $routes->post('update/(:num)', 'Employees\EmployeeController::update/$1', ['filter' => 'permission:edit-employee']);
        $routes->post('delete/(:num)', 'Employees\EmployeeController::delete/$1', ['filter' => 'permission:delete-employee']);
        $routes->get('import', 'Employees\EmployeeController::import', ['filter' => 'permission:import-employees']);
        $routes->post('import/process', 'Employees\EmployeeController::processImport', ['filter' => 'permission:import-employees']);
        $routes->get('convert-to-user/(:num)', 'Employees\EmployeeController::convertToUser/$1', ['filter' => 'permission:create-user']);
        $routes->post('process-convert-to-user', 'Employees\EmployeeController::processConvertToUser', ['filter' => 'permission:create-user']);
        $routes->get('check-user-account/(:num)', 'Employees\EmployeeController::checkUserAccount/$1');
        $routes->get('get-all-qr-codes', 'Employees\EmployeeController::getAllQRCodes');
    });

    /*
     * Attendance Management
     */
    $routes->group('attendance', ['filter' => 'permission:view-attendance'], function($routes) {
        $routes->get('', 'Attendance\AttendanceController::index');
        $routes->get('today', 'Attendance\AttendanceController::today');
        $routes->get('view/(:num)', 'Attendance\AttendanceController::view/$1');
        $routes->get('manual-entry', 'Attendance\AttendanceController::manualEntry', ['filter' => 'permission:manual-attendance']);
        $routes->post('manual-entry/save', 'Attendance\AttendanceController::saveManualEntry', ['filter' => 'permission:manual-attendance']);
        $routes->post('bulk-sign-out', 'Attendance\AttendanceController::bulkSignOut', ['filter' => 'permission:manage-attendance']);
        $routes->post('manual-sign-out/(:num)', 'Attendance\AttendanceController::manualSignOut/$1', ['filter' => 'permission:manage-attendance']);
        $routes->post('lunch-out/(:num)', 'Attendance\AttendanceController::lunchOut/$1', ['filter' => 'permission:manage-attendance']);
        $routes->post('lunch-in/(:num)', 'Attendance\AttendanceController::lunchIn/$1', ['filter' => 'permission:manage-attendance']);
        $routes->get('corrections', 'Attendance\AttendanceController::corrections');
        $routes->post('corrections/approve/(:num)', 'Attendance\AttendanceController::approveCorrection/$1', ['filter' => 'permission:approve-attendance-corrections']);
        $routes->post('corrections/reject/(:num)', 'Attendance\AttendanceController::rejectCorrection/$1', ['filter' => 'permission:approve-attendance-corrections']);
    });

    /*
     * Leave Management
     */
    $routes->group('leave', ['filter' => 'permission:view-leave-requests'], function($routes) {
        $routes->get('/', 'Leave\LeaveController::index'); // Default route for /leave
        $routes->get('requests', 'Leave\LeaveController::index');
        $routes->get('requests/view/(:num)', 'Leave\LeaveController::view/$1');
        $routes->get('requests/create', 'Leave\LeaveController::create', ['filter' => 'permission:create-leave-request']);
        $routes->post('requests/store', 'Leave\LeaveController::store', ['filter' => 'permission:create-leave-request']);
        $routes->post('requests/approve/(:num)', 'Leave\LeaveController::approve/$1', ['filter' => 'permission:approve-leave']);
        $routes->post('requests/reject/(:num)', 'Leave\LeaveController::reject/$1', ['filter' => 'permission:approve-leave']);
        $routes->get('calendar', 'Leave\LeaveController::calendar');
        $routes->get('balances', 'Leave\LeaveController::balances', ['filter' => 'permission:view-leave-balances']);
        $routes->get('types', 'Leave\LeaveController::types', ['filter' => 'permission:manage-leave-types']);
    });

    /*
     * Reports
     */
    $routes->group('reports', ['filter' => 'permission:generate-reports'], function($routes) {
        $routes->get('', 'Reports\ReportController::index');
        $routes->get('attendance', 'Reports\ReportController::attendance');
        $routes->post('attendance/generate', 'Reports\ReportController::generateAttendance');
        $routes->get('leave', 'Reports\ReportController::leave');
        $routes->post('leave/generate', 'Reports\ReportController::generateLeave');
        $routes->get('export/(:any)', 'Reports\ReportController::export/$1', ['filter' => 'permission:export-reports']);
        $routes->get('scheduled', 'Reports\ReportController::scheduled', ['filter' => 'permission:schedule-reports']);
    });

    /*
     * Settings & Administration
     */
    $routes->group('settings', function($routes) {
        // Users
        $routes->group('users', ['filter' => 'permission:view-users'], function($routes) {
            $routes->get('', 'Settings\UserController::index');
            $routes->get('view/(:num)', 'Settings\UserController::view/$1');
            $routes->get('create', 'Settings\UserController::create', ['filter' => 'permission:create-user']);
            $routes->post('store', 'Settings\UserController::store', ['filter' => 'permission:create-user']);
            $routes->get('edit/(:num)', 'Settings\UserController::edit/$1', ['filter' => 'permission:edit-user']);
            $routes->post('update/(:num)', 'Settings\UserController::update/$1', ['filter' => 'permission:edit-user']);
            $routes->post('delete/(:num)', 'Settings\UserController::delete/$1', ['filter' => 'permission:delete-user']);
            $routes->post('toggleStatus/(:num)', 'Settings\UserController::toggleStatus/$1', ['filter' => 'permission:edit-user']);
        });

        // Roles & Permissions
        $routes->group('roles', ['filter' => 'permission:manage-roles'], function($routes) {
            $routes->get('', 'Settings\RoleController::index');
            $routes->get('create', 'Settings\RoleController::create');
            $routes->post('store', 'Settings\RoleController::store');
            $routes->get('edit/(:num)', 'Settings\RoleController::edit/$1');
            $routes->post('update/(:num)', 'Settings\RoleController::update/$1');
            $routes->post('delete/(:num)', 'Settings\RoleController::delete/$1');
        });

        // Locations
        $routes->group('locations', ['filter' => 'permission:manage-locations'], function($routes) {
            $routes->get('', 'Settings\LocationController::index');
            $routes->get('create', 'Settings\LocationController::create');
            $routes->post('store', 'Settings\LocationController::store');
            $routes->get('edit/(:num)', 'Settings\LocationController::edit/$1');
            $routes->post('update/(:num)', 'Settings\LocationController::update/$1');
            $routes->post('delete/(:num)', 'Settings\LocationController::delete/$1');
        });

        // Departments
        $routes->group('departments', ['filter' => 'permission:manage-departments'], function($routes) {
            $routes->get('', 'Settings\DepartmentController::index');
            $routes->get('create', 'Settings\DepartmentController::create');
            $routes->post('store', 'Settings\DepartmentController::store');
            $routes->get('edit/(:num)', 'Settings\DepartmentController::edit/$1');
            $routes->post('update/(:num)', 'Settings\DepartmentController::update/$1');
            $routes->post('delete/(:num)', 'Settings\DepartmentController::delete/$1');
            $routes->post('toggleStatus/(:num)', 'Settings\DepartmentController::toggleStatus/$1');
        });

        // System Settings
        $routes->group('system', ['filter' => 'permission:view-system-settings'], function($routes) {
            $routes->get('', 'Settings\SystemSettingController::index');
            $routes->post('update', 'Settings\SystemSettingController::update', ['filter' => 'permission:edit-system-settings']);
        });

        // Audit Logs
        $routes->get('audit-logs', 'Settings\AuditLogController::index', ['filter' => 'permission:view-audit-logs']);
        $routes->get('audit-logs/view/(:num)', 'Settings\AuditLogController::view/$1', ['filter' => 'permission:view-audit-logs']);
        $routes->get('audit-logs/export', 'Settings\AuditLogController::export', ['filter' => 'permission:view-audit-logs']);
        $routes->post('audit-logs/clear-old', 'Settings\AuditLogController::clearOld', ['filter' => 'permission:delete-audit-logs']);

        // Sync Management
        $routes->group('sync', ['filter' => 'permission:manage-sync'], function($routes) {
            $routes->get('', 'Settings\SyncController::index');
            $routes->post('trigger/(:num)', 'Settings\SyncController::trigger/$1');
            $routes->get('logs/(:num)', 'Settings\SyncController::logs/$1');
        });
    });
});
