<?php

namespace App\Controllers\Api;

use App\Controllers\BaseController;
use App\Models\AttendanceModel;

class AttendanceController extends BaseController
{
    protected $attendanceModel;

    public function __construct()
    {
        $this->attendanceModel = new AttendanceModel();
    }

    /**
     * API Health Check
     *
     * GET /api
     * Returns 200 OK status to confirm API is accessible
     */
    public function index()
    {
        return $this->response
            ->setStatusCode(200)
            ->setContentType('application/json')
            ->setJSON([
                'status' => 'ok',
                'message' => 'Real-Time Sign-In API is running',
                'version' => '1.0.0',
                'timestamp' => date('Y-m-d H:i:s'),
                'endpoints' => [
                    'POST /api/attendance/sync' => 'Sync single offline attendance record',
                    'POST /api/attendance/sync-batch' => 'Sync multiple offline attendance records (max 100)',
                    'POST /api/attendance/validate-records' => 'Validate offline attendance data before syncing',
                    'GET /api/sync/statistics' => 'Get synchronization statistics',
                    'GET /api/reports/dashboard-stats' => 'Get dashboard statistics',
                    'GET /api/employees' => 'List all employees',
                    'GET /api/employees/{id}' => 'Get specific employee'
                ]
            ]);
    }

    /**
     * Sync offline attendance data from mobile app
     *
     * POST /api/attendance/sync
     */
    public function sync()
    {
        // Set JSON response header
        $this->response->setContentType('application/json');

        try {
            // Get JSON input
            $json = $this->request->getJSON(true);

            // Validate required fields
            if (empty($json['employee_id'])) {
                return $this->response->setStatusCode(400)->setJSON([
                    'status' => 'error',
                    'message' => 'employee_id is required'
                ]);
            }

            if (empty($json['attendance_date'])) {
                return $this->response->setStatusCode(400)->setJSON([
                    'status' => 'error',
                    'message' => 'attendance_date is required'
                ]);
            }

            // Get employee by employee_id (could be TEXT from offline DB)
            $employeeModel = new \App\Models\EmployeeModel();
            $employee = null;

            // Try to find employee by employee_id (TEXT) or employee_number
            if (is_numeric($json['employee_id'])) {
                $employee = $employeeModel->find($json['employee_id']);
            }

            if (!$employee) {
                // Try finding by employee_number if employee_id is a string
                $employee = $employeeModel->where('employee_number', $json['employee_id'])->first();
            }

            if (!$employee) {
                return $this->response->setStatusCode(400)->setJSON([
                    'status' => 'error',
                    'message' => 'Employee not found with ID: ' . $json['employee_id']
                ]);
            }

            $serverEmployeeId = $employee['employee_id'];

            // Check if record already exists using local_uuid (if provided) or employee_id + date combination
            $existingRecord = null;

            if (!empty($json['local_uuid'])) {
                $existingRecord = $this->attendanceModel
                    ->where('local_uuid', $json['local_uuid'])
                    ->first();
            }

            // Check for duplicate by employee_id + attendance_date
            if (!$existingRecord) {
                $existingRecord = $this->attendanceModel
                    ->where('employee_id', $serverEmployeeId)
                    ->where('attendance_date', $json['attendance_date'])
                    ->first();
            }

            // Prepare data for insertion/update
            $attendanceData = [
                'employee_id'     => $serverEmployeeId,
                'attendance_date' => $json['attendance_date'],
                'sign_in_time'    => $json['sign_in_time'] ?? null,
                'sign_out_time'   => $json['sign_out_time'] ?? null,
                'sign_in_method'  => $json['sign_in_method'] ?? 'Manual',
                'sign_out_method' => $json['sign_out_method'] ?? null,
                'status'          => $json['status'] ?? 'present',
                'notes'           => $json['notes'] ?? null,
            ];

            // Add optional fields if provided
            if (!empty($json['local_uuid'])) {
                $attendanceData['local_uuid'] = $json['local_uuid'];
            }

            if (!empty($json['device_time'])) {
                $attendanceData['device_time'] = $json['device_time'];
            }

            // Handle lunch times - map from offline schema
            if (!empty($json['lunch_out_time'])) {
                $attendanceData['lunch_out_time'] = $json['lunch_out_time'];
                $attendanceData['lunch_out_method'] = $json['lunch_out_method'] ?? 'Manual';
            }

            if (!empty($json['lunch_in_time'])) {
                $attendanceData['lunch_in_time'] = $json['lunch_in_time'];
                $attendanceData['lunch_in_method'] = $json['lunch_in_method'] ?? 'Manual';
            }

            // Calculate lunch_duration_minutes if both lunch times exist
            if (!empty($attendanceData['lunch_out_time']) && !empty($attendanceData['lunch_in_time'])) {
                try {
                    $lunchOutTime = new \DateTime($json['attendance_date'] . ' ' . $attendanceData['lunch_out_time']);
                    $lunchInTime = new \DateTime($json['attendance_date'] . ' ' . $attendanceData['lunch_in_time']);

                    $interval = $lunchOutTime->diff($lunchInTime);
                    $lunchDurationMinutes = ($interval->h * 60) + $interval->i;

                    $attendanceData['lunch_duration_minutes'] = $lunchDurationMinutes;
                } catch (\Exception $e) {
                    log_message('error', 'Failed to calculate lunch duration: ' . $e->getMessage());
                }
            }

            // Handle numeric fields from offline database
            if (isset($json['late_minutes'])) {
                $attendanceData['late_minutes'] = (int) $json['late_minutes'];
                $attendanceData['is_late'] = $json['late_minutes'] > 0 ? 1 : 0;
            }

            if (isset($json['working_hours'])) {
                $attendanceData['working_hours'] = (float) $json['working_hours'];
            }

            if (isset($json['overtime_hours'])) {
                $attendanceData['overtime_hours'] = (float) $json['overtime_hours'];
            }

            // Get employee's default location if not provided
            $attendanceData['location_id'] = $json['location_id'] ?? $employee['location_id'];

            if (empty($attendanceData['location_id'])) {
                return $this->response->setStatusCode(400)->setJSON([
                    'status' => 'error',
                    'message' => 'Location ID is required and employee has no default location'
                ]);
            }

            // Handle existing record - merge data intelligently
            if ($existingRecord) {
                // Merge new data with existing record (only update empty fields)
                $mergedData = $this->mergeAttendanceData($existingRecord, $attendanceData);

                // Only update if there's actually new data to merge
                if (!empty($mergedData)) {
                    // Recalculate lunch duration if lunch times were updated
                    $updatedLunchOut = $mergedData['lunch_out_time'] ?? $existingRecord['lunch_out_time'];
                    $updatedLunchIn = $mergedData['lunch_in_time'] ?? $existingRecord['lunch_in_time'];

                    if (!empty($updatedLunchOut) && !empty($updatedLunchIn)) {
                        try {
                            $lunchOutTime = new \DateTime($existingRecord['attendance_date'] . ' ' . $updatedLunchOut);
                            $lunchInTime = new \DateTime($existingRecord['attendance_date'] . ' ' . $updatedLunchIn);
                            $interval = $lunchOutTime->diff($lunchInTime);
                            $mergedData['lunch_duration_minutes'] = ($interval->h * 60) + $interval->i;
                        } catch (\Exception $e) {
                            log_message('error', 'Failed to recalculate lunch duration: ' . $e->getMessage());
                        }
                    }

                    // Recalculate working hours if sign out was just added
                    if (isset($mergedData['sign_out_time']) && !empty($existingRecord['sign_in_time'])) {
                        try {
                            $signInTime = new \DateTime($existingRecord['attendance_date'] . ' ' . $existingRecord['sign_in_time']);
                            $signOutTime = new \DateTime($existingRecord['attendance_date'] . ' ' . $mergedData['sign_out_time']);
                            $interval = $signInTime->diff($signOutTime);
                            $workingHours = $interval->h + ($interval->i / 60);
                            $mergedData['working_hours'] = round($workingHours, 2);
                        } catch (\Exception $e) {
                            log_message('error', 'Failed to calculate working hours: ' . $e->getMessage());
                        }
                    }

                    if ($this->attendanceModel->update($existingRecord['id'], $mergedData)) {
                        return $this->response->setStatusCode(200)->setJSON([
                            'status' => 'success',
                            'message' => 'Attendance record updated with new data',
                            'action' => 'updated',
                            'merged_fields' => array_keys($mergedData)
                        ]);
                    } else {
                        $errors = $this->attendanceModel->errors();
                        return $this->response->setStatusCode(500)->setJSON([
                            'status' => 'error',
                            'message' => 'Failed to update attendance',
                            'errors' => $errors
                        ]);
                    }
                } else {
                    // No new data to merge, record is already complete
                    return $this->response->setStatusCode(200)->setJSON([
                        'status' => 'success',
                        'message' => 'Record already exists with all data',
                        'action' => 'skipped'
                    ]);
                }
            }

            // Insert new attendance record
            if ($this->attendanceModel->insert($attendanceData)) {
                return $this->response->setStatusCode(201)->setJSON([
                    'status' => 'success',
                    'message' => 'Attendance synced successfully',
                    'action' => 'created',
                    'id' => $this->attendanceModel->getInsertID()
                ]);
            } else {
                $errors = $this->attendanceModel->errors();
                return $this->response->setStatusCode(500)->setJSON([
                    'status' => 'error',
                    'message' => 'Failed to sync attendance',
                    'errors' => $errors
                ]);
            }

        } catch (\Exception $e) {
            log_message('error', 'Attendance sync error: ' . $e->getMessage());

            return $this->response->setStatusCode(500)->setJSON([
                'status' => 'error',
                'message' => 'Server error occurred while syncing attendance',
                'debug' => ENVIRONMENT === 'development' ? $e->getMessage() : null
            ]);
        }
    }

    /**
     * Batch sync offline attendance records
     *
     * POST /api/attendance/sync-batch
     */
    public function syncBatch()
    {
        // Set JSON response header
        $this->response->setContentType('application/json');

        try {
            // Get JSON input
            $json = $this->request->getJSON(true);

            // Validate input is an array
            if (!isset($json['records']) || !is_array($json['records'])) {
                return $this->response->setStatusCode(400)->setJSON([
                    'status' => 'error',
                    'message' => 'records array is required'
                ]);
            }

            $records = $json['records'];

            if (empty($records)) {
                return $this->response->setStatusCode(400)->setJSON([
                    'status' => 'error',
                    'message' => 'No records provided'
                ]);
            }

            // Limit batch size to prevent overwhelming the server
            if (count($records) > 100) {
                return $this->response->setStatusCode(400)->setJSON([
                    'status' => 'error',
                    'message' => 'Batch size exceeds maximum limit of 100 records'
                ]);
            }

            $results = [
                'total' => count($records),
                'created' => 0,
                'updated' => 0,
                'failed' => 0,
                'errors' => []
            ];

            $employeeModel = new \App\Models\EmployeeModel();

            foreach ($records as $index => $record) {
                try {
                    // Validate required fields
                    if (empty($record['employee_id']) || empty($record['attendance_date'])) {
                        $results['failed']++;
                        $results['errors'][] = [
                            'index' => $index,
                            'employee_id' => $record['employee_id'] ?? 'N/A',
                            'date' => $record['attendance_date'] ?? 'N/A',
                            'message' => 'Missing required fields'
                        ];
                        continue;
                    }

                    // Get employee
                    $employee = null;
                    if (is_numeric($record['employee_id'])) {
                        $employee = $employeeModel->find($record['employee_id']);
                    }
                    if (!$employee) {
                        $employee = $employeeModel->where('employee_number', $record['employee_id'])->first();
                    }

                    if (!$employee) {
                        $results['failed']++;
                        $results['errors'][] = [
                            'index' => $index,
                            'employee_id' => $record['employee_id'],
                            'date' => $record['attendance_date'],
                            'message' => 'Employee not found'
                        ];
                        continue;
                    }

                    $serverEmployeeId = $employee['employee_id'];

                    // Check for existing record
                    $existingRecord = null;
                    if (!empty($record['local_uuid'])) {
                        $existingRecord = $this->attendanceModel
                            ->where('local_uuid', $record['local_uuid'])
                            ->first();
                    }

                    if (!$existingRecord) {
                        $existingRecord = $this->attendanceModel
                            ->where('employee_id', $serverEmployeeId)
                            ->where('attendance_date', $record['attendance_date'])
                            ->first();
                    }

                    // Prepare data
                    $attendanceData = [
                        'employee_id'     => $serverEmployeeId,
                        'attendance_date' => $record['attendance_date'],
                        'sign_in_time'    => $record['sign_in_time'] ?? null,
                        'sign_out_time'   => $record['sign_out_time'] ?? null,
                        'sign_in_method'  => $record['sign_in_method'] ?? 'Manual',
                        'sign_out_method' => $record['sign_out_method'] ?? null,
                        'status'          => $record['status'] ?? 'present',
                        'notes'           => $record['notes'] ?? null,
                        'lunch_out_time'  => $record['lunch_out_time'] ?? null,
                        'lunch_in_time'   => $record['lunch_in_time'] ?? null,
                        'late_minutes'    => isset($record['late_minutes']) ? (int) $record['late_minutes'] : 0,
                        'working_hours'   => isset($record['working_hours']) ? (float) $record['working_hours'] : null,
                        'overtime_hours'  => isset($record['overtime_hours']) ? (float) $record['overtime_hours'] : 0,
                        'location_id'     => $record['location_id'] ?? $employee['location_id'],
                    ];

                    if (!empty($record['local_uuid'])) {
                        $attendanceData['local_uuid'] = $record['local_uuid'];
                    }

                    if ($attendanceData['late_minutes'] > 0) {
                        $attendanceData['is_late'] = 1;
                    }

                    // Calculate lunch duration if needed
                    if (!empty($attendanceData['lunch_out_time']) && !empty($attendanceData['lunch_in_time'])) {
                        try {
                            $lunchOutTime = new \DateTime($record['attendance_date'] . ' ' . $attendanceData['lunch_out_time']);
                            $lunchInTime = new \DateTime($record['attendance_date'] . ' ' . $attendanceData['lunch_in_time']);
                            $interval = $lunchOutTime->diff($lunchInTime);
                            $attendanceData['lunch_duration_minutes'] = ($interval->h * 60) + $interval->i;
                        } catch (\Exception $e) {
                            log_message('error', 'Failed to calculate lunch duration: ' . $e->getMessage());
                        }
                    }

                    // Insert or merge with existing record
                    if ($existingRecord) {
                        // Merge new data with existing record
                        $mergedData = $this->mergeAttendanceData($existingRecord, $attendanceData);

                        if (!empty($mergedData)) {
                            // Recalculate lunch duration if lunch times were updated
                            $updatedLunchOut = $mergedData['lunch_out_time'] ?? $existingRecord['lunch_out_time'];
                            $updatedLunchIn = $mergedData['lunch_in_time'] ?? $existingRecord['lunch_in_time'];

                            if (!empty($updatedLunchOut) && !empty($updatedLunchIn)) {
                                try {
                                    $lunchOutTime = new \DateTime($existingRecord['attendance_date'] . ' ' . $updatedLunchOut);
                                    $lunchInTime = new \DateTime($existingRecord['attendance_date'] . ' ' . $updatedLunchIn);
                                    $interval = $lunchOutTime->diff($lunchInTime);
                                    $mergedData['lunch_duration_minutes'] = ($interval->h * 60) + $interval->i;
                                } catch (\Exception $e) {
                                    log_message('error', 'Failed to recalculate lunch duration: ' . $e->getMessage());
                                }
                            }

                            // Recalculate working hours if sign out was just added
                            if (isset($mergedData['sign_out_time']) && !empty($existingRecord['sign_in_time'])) {
                                try {
                                    $signInTime = new \DateTime($existingRecord['attendance_date'] . ' ' . $existingRecord['sign_in_time']);
                                    $signOutTime = new \DateTime($existingRecord['attendance_date'] . ' ' . $mergedData['sign_out_time']);
                                    $interval = $signInTime->diff($signOutTime);
                                    $workingHours = $interval->h + ($interval->i / 60);
                                    $mergedData['working_hours'] = round($workingHours, 2);
                                } catch (\Exception $e) {
                                    log_message('error', 'Failed to calculate working hours: ' . $e->getMessage());
                                }
                            }

                            if ($this->attendanceModel->update($existingRecord['id'], $mergedData)) {
                                $results['updated']++;
                            } else {
                                $results['failed']++;
                                $results['errors'][] = [
                                    'index' => $index,
                                    'employee_id' => $record['employee_id'],
                                    'date' => $record['attendance_date'],
                                    'message' => 'Failed to update record'
                                ];
                            }
                        } else {
                            // No new data to merge, count as skipped (still counts as successful)
                            $results['updated']++;
                        }
                    } else {
                        if ($this->attendanceModel->insert($attendanceData)) {
                            $results['created']++;
                        } else {
                            $results['failed']++;
                            $results['errors'][] = [
                                'index' => $index,
                                'employee_id' => $record['employee_id'],
                                'date' => $record['attendance_date'],
                                'message' => 'Failed to insert record',
                                'validation_errors' => $this->attendanceModel->errors()
                            ];
                        }
                    }

                } catch (\Exception $e) {
                    $results['failed']++;
                    $results['errors'][] = [
                        'index' => $index,
                        'employee_id' => $record['employee_id'] ?? 'N/A',
                        'date' => $record['attendance_date'] ?? 'N/A',
                        'message' => $e->getMessage()
                    ];
                    log_message('error', 'Batch sync error for record ' . $index . ': ' . $e->getMessage());
                }
            }

            $statusCode = $results['failed'] === 0 ? 200 : ($results['created'] + $results['updated'] > 0 ? 207 : 400);

            return $this->response->setStatusCode($statusCode)->setJSON([
                'status' => $results['failed'] === 0 ? 'success' : 'partial',
                'message' => sprintf(
                    'Processed %d records: %d created, %d updated, %d failed',
                    $results['total'],
                    $results['created'],
                    $results['updated'],
                    $results['failed']
                ),
                'results' => $results
            ]);

        } catch (\Exception $e) {
            log_message('error', 'Batch sync error: ' . $e->getMessage());

            return $this->response->setStatusCode(500)->setJSON([
                'status' => 'error',
                'message' => 'Server error occurred during batch sync',
                'debug' => ENVIRONMENT === 'development' ? $e->getMessage() : null
            ]);
        }
    }

    /**
     * Validate offline attendance data before sync
     *
     * POST /api/attendance/validate
     */
    public function validateRecords()
    {
        // Set JSON response header
        $this->response->setContentType('application/json');

        try {
            // Get JSON input
            $json = $this->request->getJSON(true);

            if (!isset($json['records']) || !is_array($json['records'])) {
                return $this->response->setStatusCode(400)->setJSON([
                    'status' => 'error',
                    'message' => 'records array is required'
                ]);
            }

            $records = $json['records'];
            $validationResults = [
                'total' => count($records),
                'valid' => 0,
                'invalid' => 0,
                'issues' => []
            ];

            $employeeModel = new \App\Models\EmployeeModel();

            foreach ($records as $index => $record) {
                $recordIssues = [];

                // Check required fields
                if (empty($record['employee_id'])) {
                    $recordIssues[] = 'employee_id is required';
                }

                if (empty($record['attendance_date'])) {
                    $recordIssues[] = 'attendance_date is required';
                } elseif (!strtotime($record['attendance_date'])) {
                    $recordIssues[] = 'attendance_date is not a valid date';
                }

                // Validate employee exists
                if (!empty($record['employee_id'])) {
                    $employee = null;
                    if (is_numeric($record['employee_id'])) {
                        $employee = $employeeModel->find($record['employee_id']);
                    }
                    if (!$employee) {
                        $employee = $employeeModel->where('employee_number', $record['employee_id'])->first();
                    }

                    if (!$employee) {
                        $recordIssues[] = 'Employee not found';
                    }
                }

                // Validate time formats
                if (!empty($record['sign_in_time']) && !$this->isValidTime($record['sign_in_time'])) {
                    $recordIssues[] = 'sign_in_time has invalid format (expected HH:MM:SS)';
                }

                if (!empty($record['sign_out_time']) && !$this->isValidTime($record['sign_out_time'])) {
                    $recordIssues[] = 'sign_out_time has invalid format (expected HH:MM:SS)';
                }

                if (!empty($record['lunch_out_time']) && !$this->isValidTime($record['lunch_out_time'])) {
                    $recordIssues[] = 'lunch_out_time has invalid format (expected HH:MM:SS)';
                }

                if (!empty($record['lunch_in_time']) && !$this->isValidTime($record['lunch_in_time'])) {
                    $recordIssues[] = 'lunch_in_time has invalid format (expected HH:MM:SS)';
                }

                // Validate status values
                $validStatuses = ['present', 'absent', 'late', 'half-day', 'on-leave'];
                if (!empty($record['status']) && !in_array($record['status'], $validStatuses)) {
                    $recordIssues[] = 'Invalid status value. Must be one of: ' . implode(', ', $validStatuses);
                }

                // Validate numeric fields
                if (isset($record['late_minutes']) && !is_numeric($record['late_minutes'])) {
                    $recordIssues[] = 'late_minutes must be numeric';
                }

                if (isset($record['working_hours']) && !is_numeric($record['working_hours'])) {
                    $recordIssues[] = 'working_hours must be numeric';
                }

                if (isset($record['overtime_hours']) && !is_numeric($record['overtime_hours'])) {
                    $recordIssues[] = 'overtime_hours must be numeric';
                }

                // Add to results
                if (empty($recordIssues)) {
                    $validationResults['valid']++;
                } else {
                    $validationResults['invalid']++;
                    $validationResults['issues'][] = [
                        'index' => $index,
                        'employee_id' => $record['employee_id'] ?? 'N/A',
                        'attendance_date' => $record['attendance_date'] ?? 'N/A',
                        'errors' => $recordIssues
                    ];
                }
            }

            return $this->response->setStatusCode(200)->setJSON([
                'status' => 'success',
                'message' => sprintf(
                    'Validated %d records: %d valid, %d invalid',
                    $validationResults['total'],
                    $validationResults['valid'],
                    $validationResults['invalid']
                ),
                'results' => $validationResults
            ]);

        } catch (\Exception $e) {
            log_message('error', 'Validation error: ' . $e->getMessage());

            return $this->response->setStatusCode(500)->setJSON([
                'status' => 'error',
                'message' => 'Server error occurred during validation',
                'debug' => ENVIRONMENT === 'development' ? $e->getMessage() : null
            ]);
        }
    }

    /**
     * Helper function to validate time format (HH:MM:SS or HH:MM)
     */
    private function isValidTime($time)
    {
        if (empty($time)) {
            return false;
        }

        // Accept HH:MM:SS or HH:MM format
        return preg_match('/^([0-1][0-9]|2[0-3]):([0-5][0-9])(:([0-5][0-9]))?$/', $time) === 1;
    }

    /**
     * Merge new attendance data with existing record
     * Only updates fields that are empty in the existing record or when new data is provided
     *
     * @param array $existingRecord Current database record
     * @param array $newData New data from offline sync
     * @return array Merged data
     */
    private function mergeAttendanceData($existingRecord, $newData)
    {
        $mergedData = [];

        // Fields that should be merged (only update if existing is null/empty)
        $mergeableFields = [
            'sign_in_time',
            'sign_out_time',
            'lunch_out_time',
            'lunch_in_time',
            'sign_in_method',
            'sign_out_method',
            'lunch_out_method',
            'lunch_in_method',
        ];

        // Fields that should always be updated if provided
        $alwaysUpdateFields = [
            'status',
            'late_minutes',
            'working_hours',
            'overtime_hours',
            'notes',
            'local_uuid',
            'device_time',
            'is_late',
            'lunch_duration_minutes'
        ];

        // Merge fields - only update if existing field is empty
        foreach ($mergeableFields as $field) {
            if (isset($newData[$field]) && !empty($newData[$field])) {
                // Update if existing is empty or new data provided
                if (empty($existingRecord[$field]) || $existingRecord[$field] === null) {
                    $mergedData[$field] = $newData[$field];
                }
                // If both exist, keep the existing value (don't overwrite)
            }
        }

        // Always update these fields if provided in new data
        foreach ($alwaysUpdateFields as $field) {
            if (isset($newData[$field])) {
                // Only update if new data has a value or is explicitly set
                if ($newData[$field] !== null && $newData[$field] !== '') {
                    $mergedData[$field] = $newData[$field];
                }
            }
        }

        return $mergedData;
    }

    /**
     * Get synchronization statistics
     *
     * GET /api/sync/statistics
     */
    public function statistics()
    {
        // Set JSON response header
        $this->response->setContentType('application/json');

        try {
            $db = \Config\Database::connect();

            // Get total synced records in last 30 days
            $thirtyDaysAgo = date('Y-m-d', strtotime('-30 days'));

            $totalRecords = $db->table('attendance')
                ->where('local_uuid IS NOT NULL')
                ->where('created_at >=', $thirtyDaysAgo)
                ->countAllResults();

            // Get total attendance records (for comparison)
            $allRecords = $db->table('attendance')
                ->where('created_at >=', $thirtyDaysAgo)
                ->countAllResults();

            // Calculate success rate
            $successRate = $allRecords > 0 ? round(($totalRecords / $allRecords) * 100, 1) : 100;

            // Get pending records (records with conflicts or needing manual review)
            // For now, we'll just return 0 as we don't have a conflicts table yet
            $pendingRecords = 0;

            return $this->response->setStatusCode(200)->setJSON([
                'status' => 'success',
                'data' => [
                    'total_records' => $totalRecords,
                    'success_rate' => $successRate,
                    'pending_records' => $pendingRecords,
                    'period' => '30 days'
                ]
            ]);

        } catch (\Exception $e) {
            log_message('error', 'Sync statistics error: ' . $e->getMessage());

            return $this->response->setStatusCode(500)->setJSON([
                'status' => 'error',
                'message' => 'Server error occurred while fetching statistics'
            ]);
        }
    }
}
