<?php

namespace App\Controllers\Dashboard;

use App\Controllers\BaseController;
use App\Models\EmployeeModel;
use App\Models\AttendanceModel;
use App\Models\LeaveRequestModel;
use App\Models\UserModel;
use App\Models\AuditLogModel;

class DashboardController extends BaseController
{
    protected $employeeModel;
    protected $attendanceModel;
    protected $leaveRequestModel;
    protected $userModel;
    protected $auditLogModel;

    public function __construct()
    {
        $this->employeeModel = new EmployeeModel();
        $this->attendanceModel = new AttendanceModel();
        $this->leaveRequestModel = new LeaveRequestModel();
        $this->userModel = new UserModel();
        $this->auditLogModel = new AuditLogModel();

        helper('auth');
    }

    /**
     * Main dashboard view
     */
    public function index()
    {
        // Get current user data
        $userId = user_id();
        $user = current_user();
        $locationId = user_location_id();

        // Get today's date
        $today = date('Y-m-d');

        // Gather dashboard statistics
        $data = [
            'pageTitle' => 'Dashboard',
            'user' => $user,

            // Employee Statistics
            'totalEmployees' => $this->getTotalEmployees($locationId),
            'activeEmployees' => $this->getActiveEmployees($locationId),
            'newEmployeesThisMonth' => $this->getNewEmployeesThisMonth($locationId),

            // Today's Attendance Statistics
            'todayStats' => $this->attendanceModel->getAttendanceStats($today, $locationId),
            'presentCount' => $this->getPresentCount($today, $locationId),
            'absentCount' => $this->getAbsentCount($today, $locationId),
            'lateCount' => $this->getLateCount($today, $locationId),
            'onLeaveCount' => $this->getOnLeaveCount($today, $locationId),

            // Leave Statistics
            'pendingLeaveRequests' => $this->getPendingLeaveRequests($locationId),
            'approvedLeavesToday' => $this->getApprovedLeavesToday($today, $locationId),

            // Recent Activities
            'recentActivities' => $this->getRecentActivities($locationId),

            // Attendance Trends (Last 7 days)
            'attendanceTrends' => $this->getAttendanceTrends($locationId),

            // Quick Stats for Charts
            'departmentBreakdown' => $this->getDepartmentBreakdown($locationId),
            'attendanceByStatus' => $this->getAttendanceByStatus($today, $locationId),
        ];

        return view('dashboard/index', $data);
    }

    /**
     * User profile view
     */
    public function profile()
    {
        $userId = user_id();
        $user = $this->userModel->getUserWithRole($userId);

        $data = [
            'pageTitle' => 'My Profile',
            'user' => $user,
        ];

        return view('dashboard/profile', $data);
    }

    /**
     * Update profile
     */
    public function updateProfile()
    {
        $userId = user_id();

        $rules = [
            'full_name' => 'required|min_length[3]|max_length[100]',
            'email' => "required|valid_email|is_unique[users.email,id,{$userId}]",
            'phone' => 'permit_empty|min_length[10]|max_length[20]',
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        $updateData = [
            'full_name' => $this->request->getPost('full_name'),
            'email' => $this->request->getPost('email'),
            'phone' => $this->request->getPost('phone'),
        ];

        // Handle profile photo upload
        $photo = $this->request->getFile('photo');
        if ($photo && $photo->isValid() && !$photo->hasMoved()) {
            $newName = $photo->getRandomName();
            $photo->move(ROOTPATH . 'public/uploads/profiles', $newName);
            $updateData['photo'] = $newName;
        }

        if ($this->userModel->update($userId, $updateData)) {
            // Log the action
            $this->auditLogModel->logAction(
                $userId,
                'update',
                'users',
                $userId,
                json_encode($updateData),
                json_encode($this->userModel->find($userId))
            );

            return redirect()->to('/profile')->with('success', 'Profile updated successfully');
        } else {
            return redirect()->back()->withInput()->with('error', 'Failed to update profile');
        }
    }

    // ========== Helper Methods for Statistics ==========

    /**
     * Get total employees count
     */
    private function getTotalEmployees($locationId = null)
    {
        $builder = $this->employeeModel->builder();

        if ($locationId && !is_super_admin()) {
            $builder->where('location_id', $locationId);
        }

        return $builder->countAllResults();
    }

    /**
     * Get active employees count
     */
    private function getActiveEmployees($locationId = null)
    {
        $builder = $this->employeeModel->builder();
        $builder->where('status', 'active');

        if ($locationId && !is_super_admin()) {
            $builder->where('location_id', $locationId);
        }

        return $builder->countAllResults();
    }

    /**
     * Get new employees this month
     */
    private function getNewEmployeesThisMonth($locationId = null)
    {
        $builder = $this->employeeModel->builder();
        $builder->where('YEAR(created_at)', date('Y'))
                ->where('MONTH(created_at)', date('m'));

        if ($locationId && !is_super_admin()) {
            $builder->where('location_id', $locationId);
        }

        return $builder->countAllResults();
    }

    /**
     * Get present count for today
     */
    private function getPresentCount($date, $locationId = null)
    {
        $builder = $this->attendanceModel->builder();
        $builder->where('attendance_date', $date)
                ->where('status', 'present');

        if ($locationId && !is_super_admin()) {
            $builder->join('employees', 'employees.employee_id = attendance.employee_id')
                    ->where('employees.location_id', $locationId);
        }

        return $builder->countAllResults();
    }

    /**
     * Get absent count for today
     */
    private function getAbsentCount($date, $locationId = null)
    {
        $totalEmployees = $this->getActiveEmployees($locationId);
        $presentCount = $this->getPresentCount($date, $locationId);
        $onLeaveCount = $this->getOnLeaveCount($date, $locationId);

        return $totalEmployees - $presentCount - $onLeaveCount;
    }

    /**
     * Get late count for today
     */
    private function getLateCount($date, $locationId = null)
    {
        $builder = $this->attendanceModel->builder();
        $builder->where('attendance_date', $date)
                ->where('is_late', 1);

        if ($locationId && !is_super_admin()) {
            $builder->join('employees', 'employees.employee_id = attendance.employee_id')
                    ->where('employees.location_id', $locationId);
        }

        return $builder->countAllResults();
    }

    /**
     * Get employees on leave today
     */
    private function getOnLeaveCount($date, $locationId = null)
    {
        $builder = $this->leaveRequestModel->builder();
        $builder->where('status', 'approved')
                ->where('start_date <=', $date)
                ->where('end_date >=', $date);

        if ($locationId && !is_super_admin()) {
            $builder->join('employees', 'employees.employee_id = leave_requests.employee_id')
                    ->where('employees.location_id', $locationId);
        }

        return $builder->countAllResults();
    }

    /**
     * Get pending leave requests count
     */
    private function getPendingLeaveRequests($locationId = null)
    {
        $builder = $this->leaveRequestModel->builder();
        $builder->where('status', 'pending');

        if ($locationId && !is_super_admin()) {
            $builder->join('employees', 'employees.employee_id = leave_requests.employee_id')
                    ->where('employees.location_id', $locationId);
        }

        return $builder->countAllResults();
    }

    /**
     * Get approved leaves for today
     */
    private function getApprovedLeavesToday($date, $locationId = null)
    {
        return $this->getOnLeaveCount($date, $locationId);
    }
 
    /**
     * Get recent audit log activities
     */
    private function getRecentActivities($locationId = null, $limit = 10)
    {
        $builder = $this->auditLogModel->builder();
        $builder->select('audit_logs.*, users.username, users.full_name')
                ->join('users', 'users.id = audit_logs.user_id', 'left')
                ->orderBy('audit_logs.created_at', 'DESC')
                ->limit($limit);

        return $builder->get()->getResultArray();
    }

    /**
     * Get attendance trends for last 7 days
     */
    private function getAttendanceTrends($locationId = null)
    {
        $trends = [];

        for ($i = 6; $i >= 0; $i--) {
            $date = date('Y-m-d', strtotime("-$i days"));
            $stats = $this->attendanceModel->getAttendanceStats($date, $locationId);

            $trends[] = [
                'date' => date('M d', strtotime($date)),
                'present' => $stats['present_count'] ?? 0,
                'late' => $stats['late_count'] ?? 0,
                'absent' => $stats['absent_count'] ?? 0,
            ];
        }

        return $trends;
    }

    /**
     * Get employee breakdown by department
     */
    private function getDepartmentBreakdown($locationId = null)
    {
        $builder = $this->employeeModel->builder();
        $builder->select('departments.name as department_name, COUNT(employees.employee_id) as count')
                ->join('departments', 'departments.id = employees.department_id', 'left')
                ->where('employees.status', 'active')
                ->groupBy('employees.department_id');

        if ($locationId && !is_super_admin()) {
            $builder->where('employees.location_id', $locationId);
        }

        return $builder->get()->getResultArray();
    }

    /**
     * Get attendance breakdown by status for today
     */
    private function getAttendanceByStatus($date, $locationId = null)
    {
        return [
            'present' => $this->getPresentCount($date, $locationId),
            'absent' => $this->getAbsentCount($date, $locationId),
            'late' => $this->getLateCount($date, $locationId),
            'on_leave' => $this->getOnLeaveCount($date, $locationId),
        ];
    }
}
