<?php

namespace App\Controllers\Employees;

use App\Controllers\BaseController;
use App\Models\EmployeeModel;
use App\Models\DepartmentModel;
use App\Models\LocationModel;
use App\Models\AuditLogModel;

class EmployeeController extends BaseController
{
    protected $employeeModel;
    protected $departmentModel;
    protected $locationModel;
    protected $auditLogModel;
    protected $userModel;

    public function __construct()
    {
        $this->employeeModel = new EmployeeModel();
        $this->departmentModel = new DepartmentModel();
        $this->locationModel = new LocationModel();
        $this->auditLogModel = new AuditLogModel();
        $this->userModel = new \App\Models\UserModel();

        helper(['auth', 'employee']);
    }

    /**
     * Display list of employees
     */
    public function index()
    {
        // Check permission
        if (!has_permission('view-employees')) {
            return redirect()->to('/dashboard')->with('error', 'You do not have permission to view employees');
        }

        // Get filters from query string
        $filters = [
            'location_id' => $this->request->getGet('location_id'),
            'department_id' => $this->request->getGet('department_id'),
            'employment_status' => $this->request->getGet('employment_status'),
            'status' => $this->request->getGet('status'),
            'search' => $this->request->getGet('search'),
        ];

        // Get employees with filters
        $employees = $this->employeeModel->getEmployeesWithFilters($filters);

        // Get departments and locations for filters
        $departments = $this->departmentModel->getActiveDepartments();
        $locations = $this->locationModel->getActiveLocations();

        $data = [
            'pageTitle' => 'Employees',
            'employees' => $employees,
            'departments' => $departments,
            'locations' => $locations,
            'filters' => $filters,
        ];

        return view('employees/index', $data);
    }

    /**
     * Display single employee details
     */
    public function view($id)
    {
        // Check permission
        if (!has_permission('view-employees')) {
            return redirect()->to('/dashboard')->with('error', 'You do not have permission to view employees');
        }

        $employee = $this->employeeModel->getEmployeeDetails($id);

        if (!$employee) {
            return redirect()->to('/employees')->with('error', 'Employee not found');
        }

        $data = [
            'pageTitle' => 'Employee Details',
            'employee' => $employee,
        ];

        return view('employees/view', $data);
    }

    /**
     * Show create employee form
     */
    public function create()
    {
        // Check permission
        if (!has_permission('create-employee')) {
            return redirect()->to('/employees')->with('error', 'You do not have permission to create employees');
        }

        $departments = $this->departmentModel->getActiveDepartments();
        $locations = $this->locationModel->getActiveLocations();

        // Get employee number settings
        $employeeNumberSettings = get_employee_number_settings();

        $data = [
            'pageTitle' => 'Add New Employee',
            'departments' => $departments,
            'locations' => $locations,
            'employeeNumberSettings' => $employeeNumberSettings,
            'nextEmployeeNumber' => $employeeNumberSettings['auto_generate'] ? generate_employee_number() : null,
        ];

        return view('employees/create', $data);
    }

    /**
     * Store new employee
     */
    public function store()
    {
        // Check permission
        if (!has_permission('create-employee')) {
            return redirect()->to('/employees')->with('error', 'You do not have permission to create employees');
        }

        // Get employee number settings
        $employeeNumberSettings = get_employee_number_settings();

        // Auto-generate employee number if enabled and not provided
        $employeeNumber = $this->request->getPost('employee_number');
        if ($employeeNumberSettings['auto_generate'] && empty($employeeNumber)) {
            $employeeNumber = generate_employee_number();
        }

        // Validation rules
        $rules = [
            'employee_number' => 'required|max_length[20]|is_unique[employees.employee_number]',
            'first_name' => 'required|max_length[50]',
            'last_name' => 'required|max_length[50]',
            'email' => 'permit_empty|valid_email|is_unique[employees.email]',
            'location_id' => 'required|is_natural_no_zero',
            'hire_date' => 'required|valid_date',
        ];

        // Override employee_number in POST data for validation
        $_POST['employee_number'] = $employeeNumber;

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        // Prepare data
        $employeeData = [
            'employee_number' => $employeeNumber,
            'first_name' => $this->request->getPost('first_name'),
            'middle_name' => $this->request->getPost('middle_name') ?: null,
            'last_name' => $this->request->getPost('last_name'),
            'email' => $this->request->getPost('email') ?: null,
            'phone' => $this->request->getPost('phone') ?: null,
            'date_of_birth' => $this->request->getPost('date_of_birth') ?: null,
            'gender' => $this->request->getPost('gender') ?: null,
            'address' => $this->request->getPost('address') ?: null,
            'city' => $this->request->getPost('city') ?: null,
            'province' => $this->request->getPost('province') ?: null,
            'zip_code' => $this->request->getPost('zip_code') ?: null,
            'department_id' => $this->request->getPost('department_id') ?: null,
            'location_id' => $this->request->getPost('location_id'),
            'position' => $this->request->getPost('position') ?: null,
            'employment_status' => $this->request->getPost('employment_status') ?: 'Probationary',
            'hire_date' => $this->request->getPost('hire_date'),
            'status' => $this->request->getPost('status'),
            'qr_code' => $this->generateQRCode(),
        ];

        // Handle photo upload
        $photo = $this->request->getFile('photo');
        if ($photo && $photo->isValid() && !$photo->hasMoved()) {
            $newName = $photo->getRandomName();
            $photo->move(ROOTPATH . 'public/uploads/employees', $newName);
            $employeeData['photo'] = $newName;
        }

        // Insert employee
        if ($this->employeeModel->insert($employeeData)) {
            $employeeId = $this->employeeModel->getInsertID();

            // Log action
            try {
                $this->auditLogModel->logAction(
                    user_id(),
                    'create',
                    'employees',
                    $employeeId,
                    'Created employee: ' . $employeeData['first_name'] . ' ' . $employeeData['last_name'],
                    null,
                    json_encode($employeeData)
                );
            } catch (\Exception $e) {
                log_message('error', 'Failed to log audit: ' . $e->getMessage());
            }

            return redirect()->to('/employees')->with('success', 'Employee added successfully');
        } else {
            $errors = $this->employeeModel->errors();
            $errorMessage = !empty($errors) ? implode(', ', $errors) : 'Failed to add employee';
            return redirect()->back()->withInput()->with('error', $errorMessage);
        }
    }

    /**
     * Show edit employee form
     */
    public function edit($id)
    {
        // Check permission
        if (!has_permission('edit-employee')) {
            return redirect()->to('/employees')->with('error', 'You do not have permission to edit employees');
        }

        $employee = $this->employeeModel->find($id);

        if (!$employee) {
            return redirect()->to('/employees')->with('error', 'Employee not found');
        }

        $departments = $this->departmentModel->getActiveDepartments();
        $locations = $this->locationModel->getActiveLocations();

        $data = [
            'pageTitle' => 'Edit Employee',
            'employee' => $employee,
            'departments' => $departments,
            'locations' => $locations,
        ];

        return view('employees/edit', $data);
    }

    /**
     * Update employee
     */
    public function update($id)
    {
        // Check permission
        if (!has_permission('edit-employee')) {
            return redirect()->to('/employees')->with('error', 'You do not have permission to edit employees');
        }

        $employee = $this->employeeModel->find($id);

        if (!$employee) {
            return redirect()->to('/employees')->with('error', 'Employee not found');
        }

        // Get posted email and employee number (database column is employee_number)
        $postedEmployeeNumber = trim($this->request->getPost('employee_number'));
        $postedEmail = trim($this->request->getPost('email'));

        // Validation rules with proper ignore syntax
        $rules = [
            'employee_number' => 'required|max_length[20]',
            'first_name' => 'required|max_length[50]',
            'last_name' => 'required|max_length[50]',
            'location_id' => 'required|is_natural_no_zero',
            'hire_date' => 'required|valid_date',
        ];

        // Collect validation errors
        $validationErrors = [];

        // Check employee number uniqueness manually (excluding current record)
        // Only check if the employee number has actually changed
        $currentEmployeeNumber = $employee['employee_number'] ?? '';
        if (!empty($postedEmployeeNumber) && strcasecmp($postedEmployeeNumber, $currentEmployeeNumber) !== 0) {
            $existingEmployee = $this->employeeModel->where('employee_number', $postedEmployeeNumber)
                ->where('id !=', $id)
                ->first();
            if ($existingEmployee) {
                $validationErrors['employee_number'] = 'This employee number already exists';
            }
        }

        // Check email uniqueness manually (excluding current record)
        // Only check if the email has actually changed and is not empty
        $currentEmail = $employee['email'] ?? '';
        // Convert both to lowercase for case-insensitive comparison
        $postedEmailLower = strtolower($postedEmail);
        $currentEmailLower = strtolower($currentEmail);

        if (!empty($postedEmail) && $postedEmailLower !== $currentEmailLower) {
            $existingEmail = $this->employeeModel->where('email', $postedEmail)
                ->where('id !=', $id)
                ->first();
            if ($existingEmail) {
                $validationErrors['email'] = 'This email is already registered';
            }
        }

        // If there are manual validation errors, return them
        if (!empty($validationErrors)) {
            return redirect()->back()->withInput()->with('errors', $validationErrors);
        }

        // Run standard validation
        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        // Prepare data (using correct database column names)
        $employeeData = [
            'employee_number' => $this->request->getPost('employee_number'),
            'first_name' => $this->request->getPost('first_name'),
            'middle_name' => $this->request->getPost('middle_name') ?: null,
            'last_name' => $this->request->getPost('last_name'),
            'email' => $this->request->getPost('email') ?: null,
            'phone' => $this->request->getPost('phone') ?: null,
            'date_of_birth' => $this->request->getPost('date_of_birth') ?: null,
            'gender' => $this->request->getPost('gender') ?: null,
            'address' => $this->request->getPost('address') ?: null,
            'city' => $this->request->getPost('city') ?: null,
            'province' => $this->request->getPost('province') ?: null,
            'zip_code' => $this->request->getPost('zip_code') ?: null,
            'department_id' => $this->request->getPost('department_id') ?: null,
            'location_id' => $this->request->getPost('location_id'),
            'position' => $this->request->getPost('position') ?: null,
            'employment_status' => $this->request->getPost('employment_status') ?: 'Probationary',
            'hire_date' => $this->request->getPost('hire_date'),
            'status' => $this->request->getPost('status'),

        ];

        // Handle photo upload
        $photo = $this->request->getFile('photo');
        if ($photo && $photo->isValid() && !$photo->hasMoved()) {
            // Delete old photo if exists
            if (!empty($employee['photo'])) {
                $oldPhotoPath = ROOTPATH . 'public/uploads/employees/' . $employee['photo'];
                if (file_exists($oldPhotoPath)) {
                    unlink($oldPhotoPath);
                }
            }

            $newName = $photo->getRandomName();
            $photo->move(ROOTPATH . 'public/uploads/employees', $newName);
            $employeeData['photo'] = $newName;
        }

        // Update employee 
        if ($this->employeeModel->skipValidation(true)->update($id, $employeeData)) {
            // Log action
            try {
                $this->auditLogModel->logAction(
                    user_id(),
                    'update',
                    'employees',
                    $id,
                    'Updated employee: ' . $employee['first_name'] . ' ' . $employee['last_name'],
                    json_encode($employee),
                    json_encode($employeeData)
                );
            } catch (\Exception $e) {
                // Log failed but don't break the flow
                log_message('error', 'Failed to log audit: ' . $e->getMessage());
            }

            return redirect()->to('/employees')->with('success', 'Employee updated successfully');
        } else {
            $errors = $this->employeeModel->errors();
            $errorMessage = !empty($errors) ? implode(', ', $errors) : 'Failed to update employee';
            return redirect()->back()->withInput()->with('error', $errorMessage);
        }
    }

    /**
     * Delete employee
     */
    public function delete($id)
    {
        // Check permission
        if (!has_permission('delete-employee')) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'You do not have permission to delete employees'
            ]);
        }

        $employee = $this->employeeModel->find($id);

        if (!$employee) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Employee not found'
            ]);
        }

        // Soft delete employee
        if ($this->employeeModel->delete($id)) {
            // Log action
            $this->auditLogModel->logAction(
                user_id(),
                'delete',
                'employees',
                $id,
                json_encode($employee),
                null
            );

            return $this->response->setJSON([
                'success' => true,
                'message' => 'Employee deleted successfully'
            ]);
        } else {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Failed to delete employee'
            ]);
        }
    }

    /**
     * Show import form
     */
    public function import()
    {
        // Check permission
        if (!has_permission('import-employees')) {
            return redirect()->to('/employees')->with('error', 'You do not have permission to import employees');
        }

        $data = [
            'pageTitle' => 'Import Employees',
        ];

        return view('employees/import', $data);
    }

    /**
     * Process CSV import
     */
    public function processImport()
    {
        // Check permission
        if (!has_permission('import-employees')) {
            return redirect()->to('/employees')->with('error', 'You do not have permission to import employees');
        }

        $csvFile = $this->request->getFile('csv_file');

        if (!$csvFile || !$csvFile->isValid()) {
            return redirect()->back()->with('error', 'Please upload a valid CSV file');
        }

        // Validate file type
        if ($csvFile->getClientMimeType() !== 'text/csv' && $csvFile->getClientExtension() !== 'csv') {
            return redirect()->back()->with('error', 'Only CSV files are allowed');
        }

        // Parse CSV
        $file = fopen($csvFile->getTempName(), 'r');
        $headers = fgetcsv($file);
        $employees = [];

        while (($row = fgetcsv($file)) !== false) {
            $employeeData = array_combine($headers, $row);

            // Add QR code
            $employeeData['qr_code'] = $this->generateQRCode();

            $employees[] = $employeeData;
        }

        fclose($file);

        // Import employees
        $result = $this->employeeModel->importEmployees($employees);

        // Log action
        $this->auditLogModel->logAction(
            user_id(),
            'import',
            'employees',
            null,
            null,
            json_encode(['imported' => $result['success'], 'errors' => count($result['errors'])])
        );

        if ($result['success'] > 0) {
            $message = "{$result['success']} employees imported successfully";
            if (count($result['errors']) > 0) {
                $message .= ", " . count($result['errors']) . " failed";
            }
            return redirect()->to('/employees')->with('success', $message);
        } else {
            return redirect()->back()->with('error', 'Failed to import employees')->with('import_errors', $result['errors']);
        }
    }

    /**
     * Generate unique QR code
     */
    private function generateQRCode()
    {
        do {
            $qrCode = 'EMP' . str_pad(rand(1, 999999), 6, '0', STR_PAD_LEFT);
            $exists = $this->employeeModel->where('qr_code', $qrCode)->first();
        } while ($exists);

        return $qrCode;
    }

    /**
     * Show form to convert employee to user
     */
    public function convertToUser($id)
    {
        // Check permission
        if (!has_permission('create-user')) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'You do not have permission to create users'
            ]);
        }

        $employee = $this->employeeModel->find($id);

        if (!$employee) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Employee not found'
            ]);
        }

        // Check if employee already has a user account
        $existingUser = $this->userModel->where('employee_id', $id)->first();
        if ($existingUser) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'This employee already has a user account'
            ]);
        }

        // Get roles for selection
        $roleModel = new \App\Models\RoleModel();
        $roles = $roleModel->findAll();

        return $this->response->setJSON([
            'success' => true,
            'employee' => $employee,
            'roles' => $roles,
            'suggested_username' => $this->generateUsername($employee)
        ]);
    }

    /**
     * Process employee to user conversion
     */
    public function processConvertToUser()
    {
        // Set JSON content type header
        $this->response->setContentType('application/json');

        // Check permission
        if (!has_permission('create-user')) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'You do not have permission to create users'
            ]);
        }

        $employeeId = $this->request->getPost('employee_id');
        $username = $this->request->getPost('username');
        $password = $this->request->getPost('password');
        $roleId = $this->request->getPost('role_id');

        // Validate inputs
        $rules = [
            'employee_id' => 'required|is_natural_no_zero',
            'username' => 'required|min_length[3]|max_length[100]|is_unique[users.username]',
            'password' => 'required|min_length[6]',
            'role_id' => 'required|is_natural_no_zero',
        ];

        if (!$this->validate($rules)) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $this->validator->getErrors()
            ]);
        }

        $employee = $this->employeeModel->find($employeeId);

        if (!$employee) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Employee not found'
            ]);
        }

        // Check if employee already has a user account
        $existingUser = $this->userModel->where('employee_id', $employeeId)->first();
        if ($existingUser) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'This employee already has a user account'
            ]);
        }

        // Create user account
        $userData = [
            'employee_id' => $employeeId,
            'username' => $username,
            'email' => $employee['email'],
            'password_hash' => password_hash($password, PASSWORD_DEFAULT),
            'first_name' => $employee['first_name'],
            'last_name' => $employee['last_name'],
            'role_id' => $roleId,
            'location_id' => $employee['location_id'],
            'phone' => $employee['phone'],
            'photo' => $employee['photo'] ?? null,
            'status' => "Active",
        ];

        if ($this->userModel->insert($userData)) {
            $userId = $this->userModel->getInsertID();

            // Log action
            try {
                $this->auditLogModel->logAction(
                    user_id(),
                    'create',
                    'users',
                    $userId,
                    'Converted employee ' . $employee['first_name'] . ' ' . $employee['last_name'] . ' to user account',
                    null,
                    json_encode($userData)
                );
            } catch (\Exception $e) {
                log_message('error', 'Failed to log audit: ' . $e->getMessage());
            }

            return $this->response->setJSON([
                'success' => true,
                'message' => 'Employee successfully converted to user. Login credentials have been created.',
                'user_id' => $userId
            ]);
        } else {
            $errors = $this->userModel->errors();
            $errorMessage = !empty($errors) ? implode(', ', $errors) : 'Failed to create user account';

            return $this->response->setJSON([
                'success' => false,
                'message' => $errorMessage,
                'errors' => $errors
            ]);
        }
    }

    /**
     * Generate username from employee data
     */
    private function generateUsername($employee)
    {
        // Try firstname.lastname format
        $baseUsername = strtolower($employee['first_name'] . '.' . $employee['last_name']);
        $baseUsername = preg_replace('/[^a-z0-9.]/', '', $baseUsername);

        $username = $baseUsername;
        $counter = 1;

        // Check if username exists and increment if needed
        while ($this->userModel->where('username', $username)->first()) {
            $username = $baseUsername . $counter;
            $counter++;
        }

        return $username;
    }

    /**
     * Check if employee has user account
     */
    public function checkUserAccount($id)
    {
        $user = $this->userModel->where('employee_id', $id)->first();

        return $this->response->setJSON([
            'has_account' => !empty($user),
            'user' => $user
        ]);
    }

    /**
     * Get all employees with QR codes for bulk download
     */
    public function getAllQRCodes()
    {
        // Check permission
        if (!has_permission('view-employees')) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'You do not have permission to view employees'
            ]);
        }

        // Get all active employees with QR codes
        $employees = $this->employeeModel
            ->select('employee_id, employee_number, first_name, last_name, qr_code')
            ->where('status', 'active')
            ->where('qr_code IS NOT NULL')
            ->where('qr_code !=', '')
            ->orderBy('employee_number', 'ASC')
            ->findAll();

        if (empty($employees)) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'No employees with QR codes found'
            ]);
        }

        // Format employee data for frontend
        $formattedEmployees = array_map(function($employee) {
            return [
                'employee_id' => $employee['employee_id'],
                'employee_number' => $employee['employee_number'],
                'name' => $employee['first_name'] . ' ' . $employee['last_name'],
                'qr_code' => $employee['qr_code']
            ];
        }, $employees);

        return $this->response->setJSON([
            'success' => true,
            'employees' => $formattedEmployees,
            'count' => count($formattedEmployees)
        ]);
    }
}
