<?php

namespace App\Controllers\Settings;

use App\Controllers\BaseController;
use App\Models\DepartmentModel;
use App\Models\EmployeeModel;
use App\Models\AuditLogModel;

class DepartmentController extends BaseController
{
    protected $departmentModel;
    protected $employeeModel;
    protected $auditLogModel;

    public function __construct()
    {
        $this->departmentModel = new DepartmentModel();
        $this->employeeModel = new EmployeeModel();
        $this->auditLogModel = new AuditLogModel();

        helper('auth');
    }

    /**
     * Display list of departments
     */
    public function index()
    {
        // Check permission
        if (!has_permission('manage-departments')) {
            return redirect()->to('/dashboard')->with('error', 'You do not have permission to manage departments');
        }

        // Get all departments with employee counts and head info
        $db = \Config\Database::connect();
        $builder = $db->table('departments d');
        $builder->select('d.*,
            COUNT(DISTINCT e.employee_id) as employee_count,
            CONCAT(head.first_name, " ", head.last_name) as head_name,
            head.employee_id as head_employee_code');
        $builder->join('employees e', 'e.department_id = d.id AND e.deleted_at IS NULL', 'left');
        $builder->join('employees head', 'head.employee_id = d.head_employee_id AND head.deleted_at IS NULL', 'left');
        $builder->where('d.deleted_at', null);
        $builder->groupBy('d.id');
        $builder->orderBy('d.created_at', 'DESC');
        $departments = $builder->get()->getResultArray();

        // Get today's attendance count for each department
        foreach ($departments as &$department) {
            $attendanceCount = $db->table('attendance a')
                ->join('employees e', 'e.employee_id = a.employee_id')
                ->where('e.department_id', $department['id'])
                ->where('a.attendance_date', date('Y-m-d'))
                ->where('a.status', 'present')
                ->countAllResults();
            $department['today_attendance'] = $attendanceCount;
        }

        $data = [
            'pageTitle' => 'Departments',
            'departments' => $departments,
        ];

        return view('settings/departments/index', $data);
    }

    /**
     * Show create department form
     */
    public function create()
    {
        // Check permission
        if (!has_permission('manage-departments')) {
            return redirect()->to('/dashboard')->with('error', 'You do not have permission to create departments');
        }

        // Get all employees for department head selection
        $employees = $this->employeeModel
            ->select('id, CONCAT(first_name, " ", last_name) as full_name')
            ->where('is_active', 1)
            ->orderBy('first_name', 'ASC')
            ->findAll();

        $data = [
            'pageTitle' => 'Create New Department',
            'employees' => $employees,
        ];

        return view('settings/departments/create', $data);
    }

    /**
     * Store new department
     */
    public function store()
    {
        // Check permission
        if (!has_permission('manage-departments')) {
            return redirect()->to('/dashboard')->with('error', 'You do not have permission to create departments');
        }

        // Validate input
        $validation = \Config\Services::validation();
        $validation->setRules([
            'name' => 'required|max_length[100]',
            'code' => 'required|max_length[20]|is_unique[departments.code]|alpha_dash',
            'description' => 'permit_empty|max_length[500]',
            'head_employee_id' => 'permit_empty|is_natural_no_zero',
            'is_active' => 'permit_empty|in_list[0,1]',
        ]);

        if (!$validation->withRequest($this->request)->run()) {
            return redirect()->back()->withInput()->with('errors', $validation->getErrors());
        }

        // Prepare data
        $data = [
            'name' => $this->request->getPost('name'),
            'code' => strtoupper($this->request->getPost('code')),
            'description' => $this->request->getPost('description'),
            'head_employee_id' => $this->request->getPost('head_employee_id') ?: null,
            'is_active' => $this->request->getPost('is_active') ? 1 : 0,
        ];

        // Insert department
        $departmentId = $this->departmentModel->insert($data);

        if (!$departmentId) {
            return redirect()->back()->withInput()->with('error', 'Failed to create department');
        }

        // Log activity
        $this->auditLogModel->insert([
            'user_id' => current_user('id'),
            'module' => 'Departments',
            'action' => 'create',
            'record_id' => $departmentId,
            'description' => 'Created department: ' . $data['name'],
            'ip_address' => $this->request->getIPAddress(),
            'user_agent' => $this->request->getUserAgent()->getAgentString(),
        ]);

        return redirect()->to('settings/departments')->with('success', 'Department created successfully');
    }

    /**
     * Show edit department form
     */
    public function edit($id)
    {
        // Check permission
        if (!has_permission('manage-departments')) {
            return redirect()->to('/dashboard')->with('error', 'You do not have permission to edit departments');
        }

        // Get department
        $department = $this->departmentModel->find($id);
        if (!$department) {
            return redirect()->to('settings/departments')->with('error', 'Department not found');
        }

        // Get employee count
        $employeeCount = $this->employeeModel->where('department_id', $id)->countAllResults();

        // Get all employees for department head selection
        $employees = $this->employeeModel
            ->select('employee_id, CONCAT(first_name, " ", last_name) as full_name')
            ->where('status', "active")
            ->orderBy('first_name', 'ASC')
            ->findAll();

        $data = [
            'pageTitle' => 'Edit Department',
            'department' => $department,
            'employeeCount' => $employeeCount,
            'employees' => $employees,
        ];

        return view('settings/departments/edit', $data);
    }

    /**
     * Update department
     */
    public function update($id)
    {
        // Check permission
        if (!has_permission('manage-departments')) {
            return redirect()->to('/dashboard')->with('error', 'You do not have permission to update departments');
        }

        // Check if department exists
        $department = $this->departmentModel->find($id);
        if (!$department) {
            return redirect()->to('settings/departments')->with('error', 'Department not found');
        }

        // Validate input
        $validation = \Config\Services::validation();
        $validation->setRules([
            'name' => 'required|max_length[100]',
            'code' => "required|max_length[20]|is_unique[departments.code,id,{$id}]|alpha_dash",
            'description' => 'permit_empty|max_length[500]',
            'head_employee_id' => 'permit_empty|is_natural_no_zero',
            'is_active' => 'permit_empty|in_list[0,1]',
        ]);

        if (!$validation->withRequest($this->request)->run()) {
            return redirect()->back()->withInput()->with('errors', $validation->getErrors());
        }

        // Prepare data
        $data = [
            'name' => $this->request->getPost('name'),
            'code' => strtoupper($this->request->getPost('code')),
            'description' => $this->request->getPost('description'),
            'head_employee_id' => $this->request->getPost('head_employee_id') ?: null,
            'is_active' => $this->request->getPost('is_active') ? 1 : 0,
        ];

        // Update department
        if (!$this->departmentModel->update($id, $data)) {
            return redirect()->back()->withInput()->with('error', 'Failed to update department');
        }

        // Log activity
        $this->auditLogModel->insert([
            'user_id' => current_user('id'),
            'module' => 'Departments',
            'action' => 'update',
            'record_id' => $id,
            'description' => 'Updated department: ' . $data['name'],
            'ip_address' => $this->request->getIPAddress(),
            'user_agent' => $this->request->getUserAgent()->getAgentString(),
        ]);

        return redirect()->to('settings/departments')->with('success', 'Department updated successfully');
    }

    /**
     * Delete department
     */
    public function delete($id)
    {
        // Check permission
        if (!has_permission('manage-departments')) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'You do not have permission to delete departments'
            ]);
        }

        // Check if department exists
        $department = $this->departmentModel->find($id);
        if (!$department) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Department not found'
            ]);
        }

        // Check if department has employees
        $employeeCount = $this->employeeModel->where('department_id', $id)->countAllResults();
        if ($employeeCount > 0) {
            return $this->response->setJSON([
                'success' => false,
                'message' => "Cannot delete department. {$employeeCount} employee(s) are assigned to this department. Please reassign employees before deleting."
            ]);
        }

        // Soft delete department
        if ($this->departmentModel->delete($id)) {
            // Log activity
            $this->auditLogModel->insert([
                'user_id' => current_user('id'),
                'module' => 'Departments',
                'action' => 'delete',
                'record_id' => $id,
                'description' => 'Deleted department: ' . $department['name'],
                'ip_address' => $this->request->getIPAddress(),
                'user_agent' => $this->request->getUserAgent()->getAgentString(),
            ]);

            return $this->response->setJSON([
                'success' => true,
                'message' => 'Department deleted successfully'
            ]);
        }

        return $this->response->setJSON([
            'success' => false,
            'message' => 'Failed to delete department'
        ]);
    }

    /**
     * Toggle department active status
     */
    public function toggleStatus($id)
    {
        // Check permission
        if (!has_permission('manage-departments')) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'You do not have permission to update departments'
            ]);
        }

        // Check if department exists
        $department = $this->departmentModel->find($id);
        if (!$department) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Department not found'
            ]);
        }

        // Toggle status
        $newStatus = $department['is_active'] ? 0 : 1;
        if ($this->departmentModel->update($id, ['is_active' => $newStatus])) {
            // Log activity
            $this->auditLogModel->insert([
                'user_id' => current_user('id'),
                'module' => 'Departments',
                'action' => 'update',
                'record_id' => $id,
                'description' => ($newStatus ? 'Activated' : 'Deactivated') . ' department: ' . $department['name'],
                'ip_address' => $this->request->getIPAddress(),
                'user_agent' => $this->request->getUserAgent()->getAgentString(),
            ]);

            return $this->response->setJSON([
                'success' => true,
                'message' => 'Department status updated successfully',
                'is_active' => $newStatus
            ]);
        }

        return $this->response->setJSON([
            'success' => false,
            'message' => 'Failed to update department status'
        ]);
    }
}
