<?php

namespace App\Controllers\Settings;

use App\Controllers\BaseController;
use App\Models\LocationModel;
use App\Models\EmployeeModel;
use App\Models\AuditLogModel;

class LocationController extends BaseController
{
    protected $locationModel;
    protected $employeeModel;
    protected $auditLogModel;

    public function __construct()
    {
        $this->locationModel = new LocationModel();
        $this->employeeModel = new EmployeeModel();
        $this->auditLogModel = new AuditLogModel();

        helper('auth');
    }

    /**
     * Display list of locations
     */
    public function index()
    {
        // Check permission
        if (!has_permission('manage-locations')) {
            return redirect()->to('/dashboard')->with('error', 'You do not have permission to manage locations');
        }

        // Get all locations with employee counts
        $db = \Config\Database::connect();
        $builder = $db->table('locations');
        $builder->select('locations.*, COUNT(DISTINCT employees.employee_id) as employee_count');
        $builder->join('employees', 'employees.location_id = locations.id AND employees.deleted_at IS NULL', 'left');
        $builder->where('locations.deleted_at', null);
        $builder->groupBy('locations.id');
        $builder->orderBy('locations.created_at', 'DESC');
        $locations = $builder->get()->getResultArray();

        // Get today's attendance count for each location
        foreach ($locations as &$location) {
            $attendanceCount = $db->table('attendance a')
                ->join('employees e', 'e.employee_id = a.employee_id')
                ->where('e.location_id', $location['id'])
                ->where('a.attendance_date', date('Y-m-d'))
                ->countAllResults();
            $location['today_attendance'] = $attendanceCount;
        }

        $data = [
            'pageTitle' => 'Locations',
            'locations' => $locations,
        ];

        return view('settings/locations/index', $data);
    }

    /**
     * Show create location form
     */
    public function create()
    {
        // Check permission
        if (!has_permission('manage-locations')) {
            return redirect()->to('/dashboard')->with('error', 'You do not have permission to create locations');
        }

        $data = [
            'pageTitle' => 'Create New Location',
        ];

        return view('settings/locations/create', $data);
    }

    /**
     * Store new location
     */
    public function store()
    {
        // Check permission
        if (!has_permission('manage-locations')) {
            return redirect()->to('/dashboard')->with('error', 'You do not have permission to create locations');
        }

        // Validate input
        $validation = \Config\Services::validation();
        $validation->setRules([
            'name' => 'required|max_length[100]',
            'code' => 'required|max_length[20]|is_unique[locations.code]|alpha_dash',
            'address' => 'permit_empty|max_length[255]',
            'city' => 'permit_empty|max_length[100]',
            'province' => 'permit_empty|max_length[100]',
            'contact_number' => 'permit_empty|max_length[50]',
            'email' => 'permit_empty|valid_email',
            'timezone' => 'permit_empty|max_length[50]',
            'latitude' => 'permit_empty|decimal',
            'longitude' => 'permit_empty|decimal',
            'is_active' => 'permit_empty|in_list[0,1]',
        ]);

        if (!$validation->withRequest($this->request)->run()) {
            return redirect()->back()->withInput()->with('errors', $validation->getErrors());
        }

        // Prepare data
        $data = [
            'name' => $this->request->getPost('name'),
            'code' => strtoupper($this->request->getPost('code')),
            'address' => $this->request->getPost('address'),
            'city' => $this->request->getPost('city'),
            'province' => $this->request->getPost('province'),
            'contact_number' => $this->request->getPost('contact_number'),
            'email' => $this->request->getPost('email'),
            'timezone' => $this->request->getPost('timezone'),
            'latitude' => $this->request->getPost('latitude'),
            'longitude' => $this->request->getPost('longitude'),
            'is_active' => $this->request->getPost('is_active') ? 1 : 0,
        ];

        // Insert location
        $locationId = $this->locationModel->insert($data);

        if (!$locationId) {
            return redirect()->back()->withInput()->with('error', 'Failed to create location');
        }

        // Log activity
        $this->auditLogModel->insert([
            'user_id' => current_user('id'),
            'module' => 'Locations',
            'action' => 'create',
            'record_id' => $locationId,
            'description' => 'Created location: ' . $data['name'],
            'ip_address' => $this->request->getIPAddress(),
            'user_agent' => $this->request->getUserAgent()->getAgentString(),
        ]);

        return redirect()->to('settings/locations')->with('success', 'Location created successfully');
    }

    /**
     * Show edit location form
     */
    public function edit($id)
    {
        // Check permission
        if (!has_permission('manage-locations')) {
            return redirect()->to('/dashboard')->with('error', 'You do not have permission to edit locations');
        }

        // Get location
        $location = $this->locationModel->find($id);
        if (!$location) {
            return redirect()->to('settings/locations')->with('error', 'Location not found');
        }

        // Get employee count
        $employeeCount = $this->employeeModel->where('location_id', $id)->countAllResults();

        $data = [
            'pageTitle' => 'Edit Location',
            'location' => $location,
            'employeeCount' => $employeeCount,
        ];

        return view('settings/locations/edit', $data);
    }

    /**
     * Update location
     */
    public function update($id)
    {
        // Check permission
        if (!has_permission('manage-locations')) {
            return redirect()->to('/dashboard')->with('error', 'You do not have permission to update locations');
        }

        // Check if location exists
        $location = $this->locationModel->find($id);
        if (!$location) {
            return redirect()->to('settings/locations')->with('error', 'Location not found');
        }

        // Validate input
        $validation = \Config\Services::validation();
        $validation->setRules([
            'name' => 'required|max_length[100]',
            'code' => "required|max_length[20]|is_unique[locations.code,id,{$id}]|alpha_dash",
            'address' => 'permit_empty|max_length[255]',
            'city' => 'permit_empty|max_length[100]',
            'province' => 'permit_empty|max_length[100]',
            'contact_number' => 'permit_empty|max_length[50]',
            'email' => 'permit_empty|valid_email',
            'timezone' => 'permit_empty|max_length[50]',
            'latitude' => 'permit_empty|decimal',
            'longitude' => 'permit_empty|decimal',
            'is_active' => 'permit_empty|in_list[0,1]',
        ]);

        if (!$validation->withRequest($this->request)->run()) {
            return redirect()->back()->withInput()->with('errors', $validation->getErrors());
        }

        // Prepare data
        $data = [
            'name' => $this->request->getPost('name'),
            'code' => strtoupper($this->request->getPost('code')),
            'address' => $this->request->getPost('address'),
            'city' => $this->request->getPost('city'),
            'province' => $this->request->getPost('province'),
            'contact_number' => $this->request->getPost('contact_number'),
            'email' => $this->request->getPost('email'),
            'timezone' => $this->request->getPost('timezone'),
            'latitude' => $this->request->getPost('latitude'),
            'longitude' => $this->request->getPost('longitude'),
            'is_active' => $this->request->getPost('is_active') ? 1 : 0,
        ];

        // Update location
        if (!$this->locationModel->update($id, $data)) {
            return redirect()->back()->withInput()->with('error', 'Failed to update location');
        }

        // Log activity
        $this->auditLogModel->insert([
            'user_id' => current_user('id'),
            'module' => 'Locations',
            'action' => 'update',
            'record_id' => $id,
            'description' => 'Updated location: ' . $data['name'],
            'ip_address' => $this->request->getIPAddress(),
            'user_agent' => $this->request->getUserAgent()->getAgentString(),
        ]);

        return redirect()->to('settings/locations')->with('success', 'Location updated successfully');
    }

    /**
     * Delete location
     */
    public function delete($id)
    {
        // Check permission
        if (!has_permission('manage-locations')) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'You do not have permission to delete locations'
            ]);
        }

        // Check if location exists
        $location = $this->locationModel->find($id);
        if (!$location) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Location not found'
            ]);
        }

        // Check if location has employees
        $employeeCount = $this->employeeModel->where('location_id', $id)->countAllResults();
        if ($employeeCount > 0) {
            return $this->response->setJSON([
                'success' => false,
                'message' => "Cannot delete location. {$employeeCount} employee(s) are assigned to this location. Please reassign employees before deleting."
            ]);
        }

        // Soft delete location
        if ($this->locationModel->delete($id)) {
            // Log activity
            $this->auditLogModel->insert([
                'user_id' => current_user('id'),
                'module' => 'Locations',
                'action' => 'delete',
                'record_id' => $id,
                'description' => 'Deleted location: ' . $location['name'],
                'ip_address' => $this->request->getIPAddress(),
                'user_agent' => $this->request->getUserAgent()->getAgentString(),
            ]);

            return $this->response->setJSON([
                'success' => true,
                'message' => 'Location deleted successfully'
            ]);
        }

        return $this->response->setJSON([
            'success' => false,
            'message' => 'Failed to delete location'
        ]);
    }

    /**
     * View location details (AJAX)
     */
    public function view($id)
    {
        // Check permission
        if (!has_permission('manage-locations')) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'You do not have permission to view locations'
            ]);
        }

        // Get location with stats
        $location = $this->locationModel->getLocationWithStats($id);
        if (!$location) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Location not found'
            ]);
        }

        return $this->response->setJSON([
            'success' => true,
            'location' => $location
        ]);
    }

    /**
     * Toggle location active status
     */
    public function toggleStatus($id)
    {
        // Check permission
        if (!has_permission('manage-locations')) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'You do not have permission to update locations'
            ]);
        }

        // Check if location exists
        $location = $this->locationModel->find($id);
        if (!$location) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Location not found'
            ]);
        }

        // Toggle status
        $newStatus = $location['is_active'] ? 0 : 1;
        if ($this->locationModel->update($id, ['is_active' => $newStatus])) {
            // Log activity
            $this->auditLogModel->insert([
                'user_id' => current_user('id'),
                'module' => 'Locations',
                'action' => 'update',
                'record_id' => $id,
                'description' => ($newStatus ? 'Activated' : 'Deactivated') . ' location: ' . $location['name'],
                'ip_address' => $this->request->getIPAddress(),
                'user_agent' => $this->request->getUserAgent()->getAgentString(),
            ]);

            return $this->response->setJSON([
                'success' => true,
                'message' => 'Location status updated successfully',
                'is_active' => $newStatus
            ]);
        }

        return $this->response->setJSON([
            'success' => false,
            'message' => 'Failed to update location status'
        ]);
    }
}
