<?php

namespace App\Controllers\Settings;

use App\Controllers\BaseController;
use App\Models\RoleModel;
use App\Models\PermissionModel;
use App\Models\AuditLogModel;
use App\Models\UserModel;

class RoleController extends BaseController
{
    protected $roleModel;
    protected $permissionModel;
    protected $auditLogModel;
    protected $userModel;

    public function __construct()
    {
        $this->roleModel = new RoleModel();
        $this->permissionModel = new PermissionModel();
        $this->auditLogModel = new AuditLogModel();
        $this->userModel = new UserModel();

        helper('auth');
    }

    /**
     * Display list of roles
     */
    public function index()
    {
        // Check permission
        if (!has_permission('manage-roles')) {
            return redirect()->to('/dashboard')->with('error', 'You do not have permission to manage roles');
        }

        // Get all roles with user counts
        $db = \Config\Database::connect();
        $builder = $db->table('roles');
        $builder->select('roles.*, COUNT(users.id) as user_count');
        $builder->join('users', 'users.role_id = roles.id', 'left');
        $builder->groupBy('roles.id');
        $builder->orderBy('roles.created_at', 'DESC');
        $roles = $builder->get()->getResultArray();

        // Get permission counts for each role
        foreach ($roles as &$role) {
            $permissionCount = $db->table('role_permissions')
                ->where('role_id', $role['id'])
                ->countAllResults();
            $role['permission_count'] = $permissionCount;
        }

        $data = [
            'pageTitle' => 'Roles & Permissions',
            'roles' => $roles,
        ];

        return view('settings/roles/index', $data);
    }

    /**
     * Show create role form
     */
    public function create()
    {
        // Check permission
        if (!has_permission('manage-roles')) {
            return redirect()->to('/dashboard')->with('error', 'You do not have permission to create roles');
        }

        // Get all permissions grouped by module
        $permissions = $this->permissionModel->getPermissionsByModule();

        $data = [
            'pageTitle' => 'Create New Role',
            'permissions' => $permissions,
        ];

        return view('settings/roles/create', $data);
    }

    /**
     * Store new role
     */
    public function store()
    {
        // Check permission
        if (!has_permission('manage-roles')) {
            return redirect()->to('/dashboard')->with('error', 'You do not have permission to create roles');
        }

        // Validate input
        $validation = \Config\Services::validation();
        $validation->setRules([
            'name' => 'required|max_length[50]|is_unique[roles.name]',
            'slug' => 'required|max_length[50]|is_unique[roles.slug]|alpha_dash',
            'description' => 'permit_empty|max_length[255]',
            'is_active' => 'permit_empty|in_list[0,1]',
        ]);

        if (!$validation->withRequest($this->request)->run()) {
            return redirect()->back()->withInput()->with('errors', $validation->getErrors());
        }

        // Prepare data
        $data = [
            'name' => $this->request->getPost('name'),
            'slug' => $this->request->getPost('slug'),
            'description' => $this->request->getPost('description'),
            'is_active' => $this->request->getPost('is_active') ? 1 : 0,
        ];

        // Insert role
        $roleId = $this->roleModel->insert($data);

        if (!$roleId) {
            return redirect()->back()->withInput()->with('error', 'Failed to create role');
        }

        // Assign permissions
        $permissionIds = $this->request->getPost('permissions') ?? [];
        if (!empty($permissionIds)) {
            $this->roleModel->assignPermissions($roleId, $permissionIds);
        }

        // Log activity
        $this->auditLogModel->insert([
            'user_id' => current_user('id'),
            'module' => 'Roles',
            'action' => 'create',
            'record_id' => $roleId,
            'description' => 'Created role: ' . $data['name'],
            'ip_address' => $this->request->getIPAddress(),
            'user_agent' => $this->request->getUserAgent()->getAgentString(),
        ]);

        return redirect()->to('settings/roles')->with('success', 'Role created successfully');
    }

    /**
     * Show edit role form
     */
    public function edit($id)
    {
        // Check permission
        if (!has_permission('manage-roles')) {
            return redirect()->to('/dashboard')->with('error', 'You do not have permission to edit roles');
        }

        // Get role with permissions
        $role = $this->roleModel->getRoleWithPermissions($id);
        if (!$role) {
            return redirect()->to('settings/roles')->with('error', 'Role not found');
        }

        // Get role permission IDs
        $rolePermissionIds = array_column($role['permissions'], 'id');

        // Get all permissions grouped by module
        $permissions = $this->permissionModel->getPermissionsByModule();

        // Get user count for this role
        $userCount = $this->userModel->where('role_id', $id)->countAllResults();

        $data = [
            'pageTitle' => 'Edit Role',
            'role' => $role,
            'rolePermissionIds' => $rolePermissionIds,
            'permissions' => $permissions,
            'userCount' => $userCount,
        ];

        return view('settings/roles/edit', $data);
    }

    /**
     * Update role
     */
    public function update($id)
    {
        // Check permission
        if (!has_permission('manage-roles')) {
            return redirect()->to('/dashboard')->with('error', 'You do not have permission to update roles');
        }

        // Check if role exists
        $role = $this->roleModel->find($id);
        if (!$role) {
            return redirect()->to('settings/roles')->with('error', 'Role not found');
        }

        // Prevent editing super_admin role
        if ($role['slug'] === 'super_admin') {
            return redirect()->back()->with('error', 'Cannot edit Super Admin role');
        }

        // Validate input
        $validation = \Config\Services::validation();
        $validation->setRules([
            'name' => "required|max_length[50]|is_unique[roles.name,id,{$id}]",
            'slug' => "required|max_length[50]|is_unique[roles.slug,id,{$id}]|alpha_dash",
            'description' => 'permit_empty|max_length[255]',
            'is_active' => 'permit_empty|in_list[0,1]',
        ]);

        if (!$validation->withRequest($this->request)->run()) {
            return redirect()->back()->withInput()->with('errors', $validation->getErrors());
        }

        // Prepare data
        $data = [
            'name' => $this->request->getPost('name'),
            'slug' => $this->request->getPost('slug'),
            'description' => $this->request->getPost('description'),
            'is_active' => $this->request->getPost('is_active') ? 1 : 0,
        ];

        // Update role
        if (!$this->roleModel->update($id, $data)) {
            return redirect()->back()->withInput()->with('error', 'Failed to update role');
        }

        // Update permissions
        $permissionIds = $this->request->getPost('permissions') ?? [];
        $this->roleModel->assignPermissions($id, $permissionIds);

        // Log activity
        $this->auditLogModel->insert([
            'user_id' => current_user('id'),
            'module' => 'Roles',
            'action' => 'update',
            'record_id' => $id,
            'description' => 'Updated role: ' . $data['name'],
            'ip_address' => $this->request->getIPAddress(),
            'user_agent' => $this->request->getUserAgent()->getAgentString(),
        ]);

        return redirect()->to('settings/roles')->with('success', 'Role updated successfully');
    }

    /**
     * Delete role
     */
    public function delete($id)
    {
        // Check permission
        if (!has_permission('manage-roles')) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'You do not have permission to delete roles'
            ]);
        }

        // Check if role exists
        $role = $this->roleModel->find($id);
        if (!$role) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Role not found'
            ]);
        }

        // Prevent deleting super_admin and admin roles
        if (in_array($role['slug'], ['super_admin', 'admin'])) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Cannot delete system roles (Super Admin, Admin)'
            ]);
        }

        // Check if role has users
        $userCount = $this->userModel->where('role_id', $id)->countAllResults();
        if ($userCount > 0) {
            return $this->response->setJSON([
                'success' => false,
                'message' => "Cannot delete role. {$userCount} user(s) are assigned to this role. Please reassign users before deleting."
            ]);
        }

        // Delete role permissions first
        $db = \Config\Database::connect();
        $db->table('role_permissions')->where('role_id', $id)->delete();

        // Delete role
        if ($this->roleModel->delete($id)) {
            // Log activity
            $this->auditLogModel->insert([
                'user_id' => current_user('id'),
                'module' => 'Roles',
                'action' => 'delete',
                'record_id' => $id,
                'description' => 'Deleted role: ' . $role['name'],
                'ip_address' => $this->request->getIPAddress(),
                'user_agent' => $this->request->getUserAgent()->getAgentString(),
            ]);

            return $this->response->setJSON([
                'success' => true,
                'message' => 'Role deleted successfully'
            ]);
        }

        return $this->response->setJSON([
            'success' => false,
            'message' => 'Failed to delete role'
        ]);
    }

    /**
     * View role details (AJAX)
     */
    public function view($id)
    {
        // Check permission
        if (!has_permission('manage-roles')) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'You do not have permission to view roles'
            ]);
        }

        // Get role with permissions
        $role = $this->roleModel->getRoleWithPermissions($id);
        if (!$role) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Role not found'
            ]);
        }

        // Get user count
        $userCount = $this->userModel->where('role_id', $id)->countAllResults();
        $role['user_count'] = $userCount;

        return $this->response->setJSON([
            'success' => true,
            'role' => $role
        ]);
    }

    /**
     * Toggle role active status
     */
    public function toggleStatus($id)
    {
        // Check permission
        if (!has_permission('manage-roles')) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'You do not have permission to update roles'
            ]);
        }

        // Check if role exists
        $role = $this->roleModel->find($id);
        if (!$role) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Role not found'
            ]);
        }

        // Prevent disabling super_admin role
        if ($role['slug'] === 'super_admin') {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Cannot disable Super Admin role'
            ]);
        }

        // Toggle status
        $newStatus = $role['is_active'] ? 0 : 1;
        if ($this->roleModel->update($id, ['is_active' => $newStatus])) {
            // Log activity
            $this->auditLogModel->insert([
                'user_id' => current_user('id'),
                'module' => 'Roles',
                'action' => 'update',
                'record_id' => $id,
                'description' => ($newStatus ? 'Activated' : 'Deactivated') . ' role: ' . $role['name'],
                'ip_address' => $this->request->getIPAddress(),
                'user_agent' => $this->request->getUserAgent()->getAgentString(),
            ]);

            return $this->response->setJSON([
                'success' => true,
                'message' => 'Role status updated successfully',
                'is_active' => $newStatus
            ]);
        }

        return $this->response->setJSON([
            'success' => false,
            'message' => 'Failed to update role status'
        ]);
    }
}
