<?php

namespace App\Controllers\Settings;

use App\Controllers\BaseController;
use App\Models\UserModel;
use App\Models\RoleModel;
use App\Models\LocationModel;
use App\Models\EmployeeModel;
use App\Models\AuditLogModel;

class UserController extends BaseController
{
    protected $userModel;
    protected $roleModel;
    protected $locationModel;
    protected $employeeModel;
    protected $auditLogModel;

    public function __construct()
    {
        $this->userModel = new UserModel();
        $this->roleModel = new RoleModel();
        $this->locationModel = new LocationModel();
        $this->employeeModel = new EmployeeModel();
        $this->auditLogModel = new AuditLogModel();

        helper('auth');
    }

    /**
     * Display list of users
     */
    public function index()
    {
        // Check permission
        if (!has_permission('view-users')) {
            return redirect()->to('/dashboard')->with('error', 'You do not have permission to view users');
        }

        // Get filters
        $filters = [
            'role_id' => $this->request->getGet('role_id'),
            'location_id' => $this->request->getGet('location_id'),
            'is_active' => $this->request->getGet('is_active'),
            'search' => $this->request->getGet('search'),
        ];

        // Build query
        $builder = $this->userModel->builder();
        $builder->select('users.*, roles.name as role_name, locations.name as location_name, employees.employee_id as employee_number');
        $builder->join('roles', 'roles.id = users.role_id', 'left');
        $builder->join('locations', 'locations.id = users.location_id', 'left');
        $builder->join('employees', 'employees.employee_id = users.id', 'left');

        // Apply filters
        if (!empty($filters['role_id'])) {
            $builder->where('users.role_id', $filters['role_id']);
        }

        if (!empty($filters['location_id'])) {
            $builder->where('users.location_id', $filters['location_id']);
        }

        if ($filters['is_active'] !== null && $filters['is_active'] !== '') {
            $builder->where('users.is_active', $filters['is_active']);
        }

        if (!empty($filters['search'])) {
            $builder->groupStart()
                ->like('users.username', $filters['search'])
                ->orLike('users.email', $filters['search'])
                ->orLike('users.first_name', $filters['search'])
                ->orLike('users.last_name', $filters['search'])
                ->groupEnd();
        }

        $builder->orderBy('users.created_at', 'DESC');
        $users = $builder->get()->getResultArray();

        // Get roles and locations for filters
        $roles = $this->roleModel->where('is_active', 1)->findAll();
        $locations = $this->locationModel->where('is_active', 1)->findAll();

        $data = [
            'pageTitle' => 'User Management',
            'users' => $users,
            'roles' => $roles,
            'locations' => $locations,
            'filters' => $filters,
        ];

        return view('settings/users/index', $data);
    }

    /**
     * Show create user form
     */
    public function create()
    {
        // Check permission
        if (!has_permission('create-user')) {
            return redirect()->to('/settings/users')->with('error', 'You do not have permission to create users');
        }

        // Get roles and locations
        $roles = $this->roleModel->where('is_active', 1)->findAll();
        $locations = $this->locationModel->where('is_active', 1)->findAll();

        // Get employees without user accounts
        $db = \Config\Database::connect();
        $builder = $db->table('employees');
        $builder->select('employees.employee_id, employees.employee_id, employees.first_name, employees.last_name, employees.email');
        $builder->join('users', 'users.id = employees.employee_id', 'left');
        $builder->where('users.id IS NULL');
        $builder->where('employees.status', 'active');
        $builder->orderBy('employees.first_name');
        $employees = $builder->get()->getResultArray();

        $data = [
            'pageTitle' => 'Create New User',
            'roles' => $roles,
            'locations' => $locations,
            'employees' => $employees,
        ];

        return view('settings/users/create', $data);
    }

    /**
     * Store new user
     */
    public function store()
    {
        // Check permission
        if (!has_permission('create-user')) {
            return redirect()->to('/settings/users')->with('error', 'You do not have permission to create users');
        }

        // Validation rules
        $rules = [
            'username' => 'required|min_length[3]|max_length[100]|is_unique[users.username]|alpha_dash',
            'email' => 'required|valid_email|max_length[255]|is_unique[users.email]',
            'password' => 'required|min_length[6]',
            'password_confirm' => 'required|matches[password]',
            'first_name' => 'required|max_length[100]',
            'last_name' => 'required|max_length[100]',
            'role_id' => 'required|is_natural_no_zero',
            'location_id' => 'permit_empty|is_natural_no_zero',
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        // Prepare data
        $userData = [
            'employee_id' => $this->request->getPost('employee_id') ?: null,
            'username' => $this->request->getPost('username'),
            'email' => $this->request->getPost('email'),
            'password_hash' => $this->request->getPost('password'),
            'first_name' => $this->request->getPost('first_name'),
            'last_name' => $this->request->getPost('last_name'),
            'role_id' => $this->request->getPost('role_id'),
            'location_id' => $this->request->getPost('location_id') ?: null,
            'phone' => $this->request->getPost('phone') ?: null,
            'is_active' => $this->request->getPost('is_active') ? 1 : 0,
        ];

        // Handle profile photo upload
        $photo = $this->request->getFile('profile_photo');
        if ($photo && $photo->isValid() && !$photo->hasMoved()) {
            $newName = $photo->getRandomName();
            $photo->move(ROOTPATH . 'public/uploads/users', $newName);
            $userData['profile_photo'] = $newName;
        }

        // Insert user
        if ($this->userModel->insert($userData)) {
            $userId = $this->userModel->getInsertID();

            // Log action
            try {
                $this->auditLogModel->insert([
                    'user_id' => user_id(),
                    'module' => 'Users',
                    'action' => 'create',
                    'record_id' => $userId,
                    'description' => 'Created user: ' . $userData['username'],
                    'ip_address' => $this->request->getIPAddress(),
                    'user_agent' => $this->request->getUserAgent()->getAgentString(),
                ]);
            } catch (\Exception $e) {
                log_message('error', 'Failed to log audit: ' . $e->getMessage());
            }

            return redirect()->to('/settings/users')->with('success', 'User created successfully');
        } else {
            $errors = $this->userModel->errors();
            $errorMessage = !empty($errors) ? implode(', ', $errors) : 'Failed to create user';
            return redirect()->back()->withInput()->with('error', $errorMessage);
        }
    }

    /**
     * Show edit user form
     */
    public function edit($id)
    {
        // Check permission
        if (!has_permission('edit-user')) {
            return redirect()->to('/settings/users')->with('error', 'You do not have permission to edit users');
        }

        // Get user
        $user = $this->userModel->find($id);
        if (!$user) {
            return redirect()->to('/settings/users')->with('error', 'User not found');
        }

        // Get user with details
        $builder = $this->userModel->builder();
        $builder->select('users.*, roles.name as role_name, locations.name as location_name, employees.employee_id as employee_number, employees.first_name as emp_first_name, employees.last_name as emp_last_name');
        $builder->join('roles', 'roles.id = users.role_id', 'left');
        $builder->join('locations', 'locations.id = users.location_id', 'left');
        $builder->join('employees', 'employees.employee_id = users.id', 'left');
        $builder->where('users.id', $id);
        $user = $builder->get()->getRowArray();

        // Get roles and locations
        $roles = $this->roleModel->where('status', 'active')->findAll();
        $locations = $this->locationModel->where('status', 'active')->findAll();

        // Prevent editing super admin by non-super admin
        $currentUser = current_user();
        if ($user['role_id'] == 1 && $currentUser['role_id'] != 1) {
            return redirect()->to('/settings/users')->with('error', 'You cannot edit a Super Admin user');
        }

        $data = [
            'pageTitle' => 'Edit User',
            'user' => $user,
            'roles' => $roles,
            'locations' => $locations,
        ];

        return view('settings/users/edit', $data);
    }

    /**
     * Update user
     */
    public function update($id)
    {
        // Check permission
        if (!has_permission('edit-user')) {
            return redirect()->to('/settings/users')->with('error', 'You do not have permission to edit users');
        }

        // Get user
        $user = $this->userModel->find($id);
        if (!$user) {
            return redirect()->to('/settings/users')->with('error', 'User not found');
        }

        // Prevent editing super admin by non-super admin
        $currentUser = current_user();
        if ($user['role_id'] == 1 && $currentUser['role_id'] != 1) {
            return redirect()->to('/settings/users')->with('error', 'You cannot edit a Super Admin user');
        }

        // Validation rules
        $rules = [
            'username' => "required|min_length[3]|max_length[100]|is_unique[users.username,id,{$id}]|alpha_dash",
            'email' => "required|valid_email|max_length[255]|is_unique[users.email,id,{$id}]",
            'first_name' => 'required|max_length[100]',
            'last_name' => 'required|max_length[100]',
            'role_id' => 'required|is_natural_no_zero',
            'location_id' => 'permit_empty|is_natural_no_zero',
        ];

        // Only validate password if provided
        if ($this->request->getPost('password')) {
            $rules['password'] = 'min_length[6]';
            $rules['password_confirm'] = 'matches[password]';
        }

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        // Prepare data
        $userData = [
            'username' => $this->request->getPost('username'),
            'email' => $this->request->getPost('email'),
            'first_name' => $this->request->getPost('first_name'),
            'last_name' => $this->request->getPost('last_name'),
            'role_id' => $this->request->getPost('role_id'),
            'location_id' => $this->request->getPost('location_id') ?: null,
            'phone' => $this->request->getPost('phone') ?: null,
            'is_active' => $this->request->getPost('is_active') ? 1 : 0,
        ];

        // Update password if provided
        if ($this->request->getPost('password')) {
            $userData['password_hash'] = $this->request->getPost('password');
        }

        // Handle profile photo upload
        $photo = $this->request->getFile('profile_photo');
        if ($photo && $photo->isValid() && !$photo->hasMoved()) {
            // Delete old photo if exists
            if (!empty($user['profile_photo'])) {
                $oldPhotoPath = ROOTPATH . 'public/uploads/users/' . $user['profile_photo'];
                if (file_exists($oldPhotoPath)) {
                    unlink($oldPhotoPath);
                }
            }

            $newName = $photo->getRandomName();
            $photo->move(ROOTPATH . 'public/uploads/users', $newName);
            $userData['profile_photo'] = $newName;
        }

        // Update user
        if ($this->userModel->update($id, $userData)) {
            // Log action
            try {
                $this->auditLogModel->insert([
                    'user_id' => user_id(),
                    'module' => 'Users',
                    'action' => 'update',
                    'record_id' => $id,
                    'description' => 'Updated user: ' . $userData['username'],
                    'ip_address' => $this->request->getIPAddress(),
                    'user_agent' => $this->request->getUserAgent()->getAgentString(),
                ]);
            } catch (\Exception $e) {
                log_message('error', 'Failed to log audit: ' . $e->getMessage());
            }

            return redirect()->to('/settings/users')->with('success', 'User updated successfully');
        } else {
            $errors = $this->userModel->errors();
            $errorMessage = !empty($errors) ? implode(', ', $errors) : 'Failed to update user';
            return redirect()->back()->withInput()->with('error', $errorMessage);
        }
    }

    /**
     * Delete user
     */
    public function delete($id)
    {
        // Check permission
        if (!has_permission('delete-user')) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'You do not have permission to delete users'
            ]);
        }

        // Get user
        $user = $this->userModel->find($id);
        if (!$user) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'User not found'
            ]);
        }

        // Prevent deleting super admin
        if ($user['role_id'] == 1) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Cannot delete Super Admin user'
            ]);
        }

        // Prevent users from deleting themselves
        if ($id == user_id()) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'You cannot delete your own account'
            ]);
        }

        // Soft delete user
        if ($this->userModel->delete($id)) {
            // Log action
            try {
                $this->auditLogModel->insert([
                    'user_id' => user_id(),
                    'module' => 'Users',
                    'action' => 'delete',
                    'record_id' => $id,
                    'description' => 'Deleted user: ' . $user['username'],
                    'ip_address' => $this->request->getIPAddress(),
                    'user_agent' => $this->request->getUserAgent()->getAgentString(),
                ]);
            } catch (\Exception $e) {
                log_message('error', 'Failed to log audit: ' . $e->getMessage());
            }

            return $this->response->setJSON([
                'success' => true,
                'message' => 'User deleted successfully'
            ]);
        }

        return $this->response->setJSON([
            'success' => false,
            'message' => 'Failed to delete user'
        ]);
    }

    /**
     * Toggle user active status
     */
    public function toggleStatus($id)
    {
        // Check permission
        if (!has_permission('edit-user')) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'You do not have permission to update users'
            ]);
        }

        // Get user
        $user = $this->userModel->find($id);
        if (!$user) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'User not found'
            ]);
        }

        // Prevent disabling super admin
        if ($user['role_id'] == 1) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Cannot disable Super Admin user'
            ]);
        }

        // Prevent users from disabling themselves
        if ($id == user_id()) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'You cannot disable your own account'
            ]);
        }

        // Toggle status
        $newStatus = $user['is_active'] ? 0 : 1;
        if ($this->userModel->update($id, ['is_active' => $newStatus])) {
            // Log action
            try {
                $this->auditLogModel->insert([
                    'user_id' => user_id(),
                    'module' => 'Users',
                    'action' => 'update',
                    'record_id' => $id,
                    'description' => ($newStatus ? 'Activated' : 'Deactivated') . ' user: ' . $user['username'],
                    'ip_address' => $this->request->getIPAddress(),
                    'user_agent' => $this->request->getUserAgent()->getAgentString(),
                ]);
            } catch (\Exception $e) {
                log_message('error', 'Failed to log audit: ' . $e->getMessage());
            }

            return $this->response->setJSON([
                'success' => true,
                'message' => 'User status updated successfully',
                'is_active' => $newStatus
            ]);
        }

        return $this->response->setJSON([
            'success' => false,
            'message' => 'Failed to update user status'
        ]);
    }

    /**
     * View user details
     */
    public function view($id)
    {
        // Check permission
        if (!has_permission('view-users')) {
            return redirect()->to('/dashboard')->with('error', 'You do not have permission to view users');
        }

        // Get user with details
        $builder = $this->userModel->builder();
        $builder->select('users.*, roles.name as role_name, roles.slug as role_slug, locations.name as location_name, employees.employee_id as employee_number, employees.first_name as emp_first_name, employees.last_name as emp_last_name');
        $builder->join('roles', 'roles.id = users.role_id', 'left');
        $builder->join('locations', 'locations.id = users.location_id', 'left');
        $builder->join('employees', 'employees.employee_id = users.id', 'left');
        $builder->where('users.id', $id);
        $user = $builder->get()->getRowArray();

        if (!$user) {
            return redirect()->to('/settings/users')->with('error', 'User not found');
        }

        // Get user permissions
        $db = \Config\Database::connect();
        $permissions = $db->table('role_permissions')
            ->select('permissions.name, permissions.slug, permissions.module')
            ->join('permissions', 'permissions.id = role_permissions.permission_id')
            ->where('role_permissions.role_id', $user['role_id'])
            ->orderBy('permissions.module')
            ->orderBy('permissions.name')
            ->get()
            ->getResultArray();

        // Get recent activity
        $recentActivity = $this->auditLogModel
            ->where('user_id', $id)
            ->orderBy('created_at', 'DESC')
            ->limit(10)
            ->findAll();

        $data = [
            'pageTitle' => 'User Details',
            'user' => $user,
            'permissions' => $permissions,
            'recentActivity' => $recentActivity,
        ];

        return view('settings/users/view', $data);
    }
}
