<?php

if (!function_exists('generate_employee_number')) {
    /**
     * Generate next employee number based on system settings
     *
     * @return string|null
     */
    function generate_employee_number()
    {
        $settingsModel = new \App\Models\SystemSettingModel();
        $employeeModel = new \App\Models\EmployeeModel();

        // Check if auto-generation is enabled
        $autoGenerate = $settingsModel->getSetting('enable_auto_employee_number', true);

        if (!$autoGenerate) {
            return null;
        }

        // Get settings
        $prefix = $settingsModel->getSetting('employee_number_prefix', 'EMP');
        $padding = $settingsModel->getSetting('employee_number_padding', 4);
        $startNumber = $settingsModel->getSetting('employee_number_start', 1);

        // Get the last employee number with the same prefix
        $lastEmployee = $employeeModel
            ->select('employee_number')
            ->like('employee_number', $prefix, 'after')
            ->orderBy('employee_id', 'DESC')
            ->first();

        if ($lastEmployee) {
            // Extract the numeric part from the last employee number
            $lastNumber = preg_replace('/\D/', '', $lastEmployee['employee_number']);
            $nextNumber = (int)$lastNumber + 1;
        } else {
            // No employees with this prefix yet, start from the configured start number
            $nextNumber = $startNumber;
        }

        // Format the employee number with padding
        $formattedNumber = str_pad($nextNumber, $padding, '0', STR_PAD_LEFT);

        return $prefix . $formattedNumber;
    }
}

if (!function_exists('get_employee_number_settings')) {
    /**
     * Get employee number generation settings
     *
     * @return array
     */
    function get_employee_number_settings()
    {
        $settingsModel = new \App\Models\SystemSettingModel();

        return [
            'auto_generate' => $settingsModel->getSetting('enable_auto_employee_number', true),
            'prefix' => $settingsModel->getSetting('employee_number_prefix', 'EMP'),
            'padding' => $settingsModel->getSetting('employee_number_padding', 4),
            'start_number' => $settingsModel->getSetting('employee_number_start', 1),
        ];
    }
}

if (!function_exists('preview_employee_number')) {
    /**
     * Preview what the next employee number will look like
     *
     * @param string|null $prefix
     * @param int|null $padding
     * @param int|null $startNumber
     * @return string
     */
    function preview_employee_number($prefix = null, $padding = null, $startNumber = null)
    {
        $settingsModel = new \App\Models\SystemSettingModel();
        $employeeModel = new \App\Models\EmployeeModel();

        // Use provided values or get from settings
        $prefix = $prefix ?? $settingsModel->getSetting('employee_number_prefix', 'EMP');
        $padding = $padding ?? $settingsModel->getSetting('employee_number_padding', 4);
        $startNumber = $startNumber ?? $settingsModel->getSetting('employee_number_start', 1);

        // Get the last employee number with the same prefix
        $lastEmployee = $employeeModel
            ->select('employee_number')
            ->like('employee_number', $prefix, 'after')
            ->orderBy('employee_id', 'DESC')
            ->first();

        if ($lastEmployee) {
            // Extract the numeric part from the last employee number
            $lastNumber = preg_replace('/\D/', '', $lastEmployee['employee_number']);
            $nextNumber = (int)$lastNumber + 1;
        } else {
            $nextNumber = $startNumber;
        }

        // Format the employee number with padding
        $formattedNumber = str_pad($nextNumber, $padding, '0', STR_PAD_LEFT);

        return $prefix . $formattedNumber;
    }
}
