<?php

namespace App\Models;

use CodeIgniter\Model;

class EmployeeModel extends Model
{
    protected $table            = 'employees';
    protected $primaryKey       = 'employee_id';
    protected $useAutoIncrement = true;
    protected $returnType       = 'array';
    protected $useSoftDeletes   = true;
    protected $protectFields    = true;
    protected $allowedFields    = [
        'employee_id',
        'employee_number',
        'qr_code',
        'first_name',
        'middle_name',
        'last_name',
        'email',
        'phone',
        'date_of_birth',
        'gender',
        'address',
        'city',
        'province',
        'postal_code',
        'department_id',
        'location_id',
        'position',
        'employment_status',
        'hire_date',
        'termination_date',
        'is_active',
        'photo_path',
        'facial_recognition_data'
    ];

    // Dates
    protected $useTimestamps = true;
    protected $dateFormat    = 'datetime';
    protected $createdField  = 'created_at';
    protected $updatedField  = 'updated_at';
    protected $deletedField  = 'deleted_at';

    // Validation
    protected $validationRules = [
        'employee_number' => 'required|max_length[20]|is_unique[employees.employee_number,id,{id}]',
        'first_name'      => 'required|max_length[50]',
        'last_name'       => 'required|max_length[50]',
        'email'           => 'permit_empty|valid_email|is_unique[employees.email,id,{id}]',
        'location_id'     => 'required|is_natural_no_zero',
        'hire_date'       => 'required|valid_date',
    ];

    protected $validationMessages = [
        'employee_number' => [
            'required'  => 'Employee number is required',
            'is_unique' => 'This employee number already exists',
        ],
        'email' => [
            'valid_email' => 'Please enter a valid email address',
            'is_unique'   => 'This email is already registered',
        ],
    ];

    protected $skipValidation       = false;
    protected $cleanValidationRules = true;

    /**
     * Get employee with related data
     */
    public function getEmployeeDetails($id)
    {
        return $this->select('employees.*,
                departments.name as department_name,
                locations.name as location_name')
            ->join('departments', 'departments.id = employees.department_id', 'left')
            ->join('locations', 'locations.id = employees.location_id', 'left')
            ->find($id);
    }

    /**
     * Get employees with filters
     */
    public function getEmployeesWithFilters($filters = [])
    {
        $builder = $this->select('employees.*,
                departments.name as department_name,
                locations.name as location_name')
            ->join('departments', 'departments.id = employees.department_id', 'left')
            ->join('locations', 'locations.id = employees.location_id', 'left');

        if (!empty($filters['location_id'])) {
            $builder->where('employees.location_id', $filters['location_id']);
        }

        if (!empty($filters['department_id'])) {
            $builder->where('employees.department_id', $filters['department_id']);
        }

        if (!empty($filters['employment_status'])) {
            $builder->where('employees.employment_status', $filters['employment_status']);
        }

        if (!empty($filters['status'])) {
            $builder->where('employees.status', $filters['status']);
        }

        if (!empty($filters['search'])) {
            $builder->groupStart()
                ->like('employees.employee_number', $filters['search'])
                ->orLike('employees.first_name', $filters['search'])
                ->orLike('employees.last_name', $filters['search'])
                ->orLike('employees.email', $filters['search'])
                ->groupEnd();
        }

        return $builder->findAll();
    }

    /**
     * Get employee by QR code
     */
    public function getByQRCode($qrCode)
    {
        return $this->where('qr_code', $qrCode)
            ->where('is_active', 1)
            ->first();
    }

    /**
     * Import employees from CSV data
     */
    public function importEmployees($data)
    {
        $inserted = 0;
        $errors = [];

        foreach ($data as $index => $row) {
            try {
                if ($this->insert($row)) {
                    $inserted++;
                } else {
                    $errors[] = [
                        'row'    => $index + 1,
                        'errors' => $this->errors()
                    ];
                }
            } catch (\Exception $e) {
                $errors[] = [
                    'row'     => $index + 1,
                    'message' => $e->getMessage()
                ];
            }
        }

        return [
            'success' => $inserted,
            'errors'  => $errors
        ];
    }

    /**
     * Get full name
     */
    public function getFullName($employee)
    {
        $name = $employee['first_name'];
        if (!empty($employee['middle_name'])) {
            $name .= ' ' . $employee['middle_name'];
        }
        $name .= ' ' . $employee['last_name'];
        return $name;
    }
}
