<?php

namespace App\Models;

use CodeIgniter\Model;

class ReportModel extends Model
{
    protected $table            = 'scheduled_reports';
    protected $primaryKey       = 'id';
    protected $useAutoIncrement = true;
    protected $returnType       = 'array';
    protected $useSoftDeletes   = false;
    protected $protectFields    = true;
    protected $allowedFields    = [
        'report_name',
        'report_type',
        'frequency',
        'schedule_time',
        'schedule_day',
        'filter_params',
        'export_format',
        'recipients',
        'is_active',
        'last_run_date',
        'next_run_date',
        'created_by'
    ];

    // Dates
    protected $useTimestamps = true;
    protected $dateFormat    = 'datetime';
    protected $createdField  = 'created_at';
    protected $updatedField  = 'updated_at';

    // Validation
    protected $validationRules = [
        'report_name'   => 'required|max_length[255]',
        'report_type'   => 'required|in_list[attendance,leave,employee,custom]',
        'frequency'     => 'required|in_list[daily,weekly,monthly]',
        'export_format' => 'required|in_list[pdf,excel,csv]',
        'recipients'    => 'required|max_length[500]',
    ];

    protected $skipValidation       = false;
    protected $cleanValidationRules = true;

    /**
     * Get all scheduled reports with creator information
     */
    public function getScheduledReportsWithCreator()
    {
        return $this->select('scheduled_reports.*, users.full_name')
                    ->join('users', 'users.id = scheduled_reports.created_by', 'left')
                    ->orderBy('scheduled_reports.created_at', 'DESC')
                    ->findAll();
    }

    /**
     * Get active scheduled reports that are due to run
     */
    public function getDueReports()
    {
        return $this->where('is_active', 1)
                    ->where('next_run_date <=', date('Y-m-d H:i:s'))
                    ->findAll();
    }

    /**
     * Get scheduled reports by user
     */
    public function getReportsByUser($userId)
    {
        return $this->where('created_by', $userId)
                    ->orderBy('created_at', 'DESC')
                    ->findAll();
    }

    /**
     * Update last run and calculate next run date
     */
    public function updateRunDates($reportId)
    {
        $report = $this->find($reportId);
        if (!$report) {
            return false;
        }

        $now = new \DateTime();
        $nextRun = $this->calculateNextRunDate($report['frequency'], $report['schedule_time'], $report['schedule_day']);

        return $this->update($reportId, [
            'last_run_date' => $now->format('Y-m-d H:i:s'),
            'next_run_date' => $nextRun
        ]);
    }

    /**
     * Calculate next run date based on frequency
     */
    private function calculateNextRunDate($frequency, $scheduleTime, $scheduleDay = null)
    {
        $now = new \DateTime();
        $nextRun = new \DateTime();

        // Set the time from schedule_time
        if ($scheduleTime) {
            $timeParts = explode(':', $scheduleTime);
            $nextRun->setTime((int)$timeParts[0], (int)$timeParts[1], 0);
        }

        switch ($frequency) {
            case 'daily':
                // If current time has passed today's scheduled time, set to tomorrow
                if ($now > $nextRun) {
                    $nextRun->modify('+1 day');
                }
                break;

            case 'weekly':
                // Set to next occurrence of the specified day of week
                $targetDay = $scheduleDay ?? 1; // Default to Monday
                $currentDay = (int)$now->format('N');

                if ($currentDay > $targetDay || ($currentDay == $targetDay && $now > $nextRun)) {
                    $daysToAdd = 7 - $currentDay + $targetDay;
                } else {
                    $daysToAdd = $targetDay - $currentDay;
                }

                $nextRun->modify("+{$daysToAdd} days");
                break;

            case 'monthly':
                // Set to next occurrence of the specified day of month
                $targetDay = $scheduleDay ?? 1; // Default to 1st of month
                $currentDay = (int)$now->format('j');

                if ($currentDay > $targetDay || ($currentDay == $targetDay && $now > $nextRun)) {
                    // Move to next month
                    $nextRun->modify('first day of next month');
                    $nextRun->modify('+' . ($targetDay - 1) . ' days');
                } else {
                    // Same month
                    $nextRun->modify('+' . ($targetDay - $currentDay) . ' days');
                }
                break;
        }

        return $nextRun->format('Y-m-d H:i:s');
    }

    /**
     * Toggle schedule active status
     */
    public function toggleActive($reportId)
    {
        $report = $this->find($reportId);
        if (!$report) {
            return false;
        }

        $newStatus = $report['is_active'] ? 0 : 1;

        // If activating, calculate next run date
        $updateData = ['is_active' => $newStatus];
        if ($newStatus == 1) {
            $updateData['next_run_date'] = $this->calculateNextRunDate(
                $report['frequency'],
                $report['schedule_time'],
                $report['schedule_day']
            );
        }

        return $this->update($reportId, $updateData);
    }

    /**
     * Get report statistics
     */
    public function getReportStatistics($userId = null)
    {
        $builder = $this->builder();

        if ($userId) {
            $builder->where('created_by', $userId);
        }

        $stats = [
            'total' => $builder->countAllResults(false),
            'active' => $builder->where('is_active', 1)->countAllResults(false),
            'daily' => $builder->where('frequency', 'daily')->countAllResults(false),
            'weekly' => $builder->where('frequency', 'weekly')->countAllResults(false),
            'monthly' => $builder->where('frequency', 'monthly')->countAllResults(false),
        ];

        return $stats;
    }

    /**
     * Get upcoming scheduled reports (next 7 days)
     */
    public function getUpcomingReports($days = 7)
    {
        $endDate = date('Y-m-d H:i:s', strtotime("+{$days} days"));

        return $this->select('scheduled_reports.*, users.full_name')
                    ->join('users', 'users.id = scheduled_reports.created_by', 'left')
                    ->where('is_active', 1)
                    ->where('next_run_date >=', date('Y-m-d H:i:s'))
                    ->where('next_run_date <=', $endDate)
                    ->orderBy('next_run_date', 'ASC')
                    ->findAll();
    }

    /**
     * Get report execution history (using audit logs or separate table)
     */
    public function getReportHistory($reportId, $limit = 10)
    {
        // This would typically query a separate report_history table
        // For now, returning placeholder structure
        return [];
    }

    /**
     * Create a new scheduled report with next run date
     */
    public function createScheduledReport($data)
    {
        // Calculate next run date
        if (!isset($data['next_run_date'])) {
            $data['next_run_date'] = $this->calculateNextRunDate(
                $data['frequency'],
                $data['schedule_time'] ?? '09:00:00',
                $data['schedule_day'] ?? null
            );
        }

        return $this->insert($data);
    }

    /**
     * Update scheduled report
     */
    public function updateScheduledReport($reportId, $data)
    {
        // Recalculate next run date if frequency or schedule changed
        if (isset($data['frequency']) || isset($data['schedule_time']) || isset($data['schedule_day'])) {
            $report = $this->find($reportId);
            $data['next_run_date'] = $this->calculateNextRunDate(
                $data['frequency'] ?? $report['frequency'],
                $data['schedule_time'] ?? $report['schedule_time'],
                $data['schedule_day'] ?? $report['schedule_day']
            );
        }

        return $this->update($reportId, $data);
    }

    /**
     * Delete scheduled report
     */
    public function deleteScheduledReport($reportId)
    {
        return $this->delete($reportId);
    }

    /**
     * Get reports by type
     */
    public function getReportsByType($reportType)
    {
        return $this->where('report_type', $reportType)
                    ->where('is_active', 1)
                    ->findAll();
    }

    /**
     * Get reports by frequency
     */
    public function getReportsByFrequency($frequency)
    {
        return $this->where('frequency', $frequency)
                    ->where('is_active', 1)
                    ->findAll();
    }

    /**
     * Check if report is due to run
     */
    public function isReportDue($reportId)
    {
        $report = $this->find($reportId);
        if (!$report || !$report['is_active']) {
            return false;
        }

        return strtotime($report['next_run_date']) <= time();
    }
}
