<?= $this->extend('layouts/admin') ?>

<?= $this->section('content') ?>

<!-- Header with Actions -->
<div class="d-flex justify-content-between align-items-center mb-4">
    <h2><i class="fas fa-users me-2"></i><?= esc($pageTitle) ?></h2>
    <div>
        <?php if (has_permission('view-employees')): ?>
        <button type="button" class="btn btn-outline-success me-2" id="downloadAllQRBtn">
            <i class="fas fa-qrcode"></i> Download All QR Codes
        </button>
        <?php endif; ?>

        <?php if (has_permission('import-employees')): ?>
        <a href="<?= base_url('employees/import') ?>" class="btn btn-outline-primary me-2">
            <i class="fas fa-file-import"></i> Import
        </a>
        <?php endif; ?>

        <?php if (has_permission('create-employee')): ?>
        <a href="<?= base_url('employees/create') ?>" class="btn btn-primary">
            <i class="fas fa-plus"></i> Add Employee
        </a>
        <?php endif; ?>
    </div>
</div>

<!-- Filters -->
<div class="table-card mb-4">
    <form method="get" action="<?= base_url('employees') ?>" class="row g-3">
        <div class="col-md-3">
            <label class="form-label">Search</label>
            <input type="text" name="search" class="form-control" placeholder="Employee name, number, email..." value="<?= esc($filters['search'] ?? '') ?>">
        </div>

        <div class="col-md-2">
            <label class="form-label">Location</label>
            <select name="location_id" class="form-select">
                <option value="">All Locations</option>
                <?php foreach ($locations as $location): ?>
                <option value="<?= $location['id'] ?>" <?= ($filters['location_id'] ?? '') == $location['id'] ? 'selected' : '' ?>>
                    <?= esc($location['name']) ?>
                </option>
                <?php endforeach; ?>
            </select>
        </div>

        <div class="col-md-2">
            <label class="form-label">Department</label>
            <select name="department_id" class="form-select">
                <option value="">All Departments</option>
                <?php foreach ($departments as $department): ?>
                <option value="<?= $department['id'] ?>" <?= ($filters['department_id'] ?? '') == $department['id'] ? 'selected' : '' ?>>
                    <?= esc($department['name']) ?>
                </option>
                <?php endforeach; ?>
            </select>
        </div>

        <div class="col-md-2">
            <label class="form-label">Employment Status</label>
            <select name="employment_status" class="form-select">
                <option value="">All Types</option>
                <option value="full-time" <?= ($filters['employment_status'] ?? '') == 'full-time' ? 'selected' : '' ?>>Full-Time</option>
                <option value="part-time" <?= ($filters['employment_status'] ?? '') == 'part-time' ? 'selected' : '' ?>>Part-Time</option>
                <option value="contract" <?= ($filters['employment_status'] ?? '') == 'contract' ? 'selected' : '' ?>>Contract</option>
            </select>
        </div>

        <div class="col-md-2">
            <label class="form-label">Status</label>
            <select name="status" class="form-select">
                <option value="">All Status</option>
                <option value="active" <?= ($filters['status'] ?? '') == 'active' ? 'selected' : '' ?>>Active</option>
                <option value="inactive" <?= ($filters['status'] ?? '') == 'inactive' ? 'selected' : '' ?>>Inactive</option>
            </select>
        </div>

        <div class="col-md-1 d-flex align-items-end">
            <button type="submit" class="btn btn-primary w-100">
                <i class="fas fa-filter"></i> Filter
            </button>
        </div>
    </form>
</div>

<!-- Flash Messages -->
<?php if (session()->has('success')): ?>
<div class="alert alert-success alert-dismissible fade show" role="alert">
    <i class="fas fa-check-circle me-2"></i><?= session('success') ?>
    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
</div>
<?php endif; ?>

<?php if (session()->has('error')): ?>
<div class="alert alert-danger alert-dismissible fade show" role="alert">
    <i class="fas fa-exclamation-circle me-2"></i><?= session('error') ?>
    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
</div>
<?php endif; ?>

<!-- Employees Table -->
<div class="table-card">
    <div class="table-responsive">
        <table class="table table-hover align-middle">
            <thead>
                <tr>
                    <th>Photo</th>
                    <th>Employee #</th>
                    <th>Name</th>
                    <th>Email</th>
                    <th>Department</th>
                    <th>Location</th>
                    <th>Position</th>
                    <th>Employment</th>
                    <th>Status</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php if (!empty($employees)): ?>
                    <?php foreach ($employees as $employee): ?>
                    <tr>
                        <td>
                            <?php if (!empty($employee['photo'])): ?>
                            <img src="<?= base_url('uploads/employees/' . $employee['photo']) ?>"
                                 alt="<?= esc($employee['first_name']) ?>"
                                 class="rounded-circle"
                                 style="width: 40px; height: 40px; object-fit: cover;">
                            <?php else: ?>
                            <div class="rounded-circle bg-secondary text-white d-flex align-items-center justify-content-center"
                                 style="width: 40px; height: 40px;">
                                <i class="fas fa-user"></i>
                            </div>
                            <?php endif; ?>
                        </td>
                        <td><strong><?= esc($employee['employee_number']) ?></strong></td>
                        <td>
                            <?= esc($employee['first_name'] . ' ' . $employee['last_name']) ?>
                            <?php if (!empty($employee['qr_code'])): ?>
                            <br><small class="text-muted">QR: <?= esc($employee['qr_code']) ?></small>
                            <?php endif; ?>
                        </td>
                        <td><?= esc($employee['email'] ?? '-') ?></td>
                        <td><?= esc($employee['department_name'] ?? '-') ?></td>
                        <td><?= esc($employee['location_name'] ?? '-') ?></td>
                        <td><?= esc($employee['position'] ?? '-') ?></td>
                        <td>
                            <span class="badge bg-<?= $employee['employment_status'] == 'full-time' ? 'primary' : ($employee['employment_status'] == 'part-time' ? 'info' : 'secondary') ?>">
                                <?= esc(ucwords(str_replace('-', ' ', $employee['employment_status'] ?? 'N/A'))) ?>
                            </span>
                        </td>
                        <td>
                            <span class="badge bg-<?= $employee['status'] == 'active' ? 'success' : 'danger' ?>">
                                <?= esc(ucfirst($employee['status'])) ?>
                            </span>
                        </td>
                        <td>
                            <div class="btn-group btn-group-sm" role="group">
                                <?php if (has_permission('view-employees')): ?>
                                <a href="<?= base_url('employees/view/' . $employee['employee_id']) ?>"
                                   class="btn btn-outline-info"
                                   title="View Details">
                                    <i class="fas fa-eye"></i>
                                </a>
                                <?php endif; ?>

                                <?php if (has_permission('edit-employee')): ?>
                                <a href="<?= base_url('employees/edit/' . $employee['employee_id']) ?>"
                                   class="btn btn-outline-primary"
                                   title="Edit">
                                    <i class="fas fa-edit"></i>
                                </a>
                                <?php endif; ?>

                                <?php if (has_permission('delete-employee')): ?>
                                <button type="button"
                                        class="btn btn-outline-danger delete-btn"
                                        data-id="<?= $employee['employee_id'] ?>"
                                        data-name="<?= esc($employee['first_name'] . ' ' . $employee['last_name']) ?>"
                                        title="Delete">
                                    <i class="fas fa-trash"></i>
                                </button>
                                <?php endif; ?>
                            </div>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                <?php else: ?>
                    <tr>
                        <td colspan="10" class="text-center py-5 text-muted">
                            <i class="fas fa-users fa-3x mb-3"></i>
                            <p>No employees found</p>
                        </td>
                    </tr>
                <?php endif; ?>
            </tbody>
        </table>
    </div>

    <?php if (!empty($employees)): ?>
    <div class="p-3 border-top">
        <div class="d-flex justify-content-between align-items-center">
            <div class="text-muted">
                Showing <?= count($employees) ?> employee(s)
            </div>
        </div>
    </div>
    <?php endif; ?>
</div>

<?= $this->endSection() ?>

<?= $this->section('scripts') ?>
<script>
// Delete employee confirmation
document.querySelectorAll('.delete-btn').forEach(button => {
    button.addEventListener('click', function() {
        const employeeId = this.dataset.id;
        const employeeName = this.dataset.name;

        if (confirm(`Are you sure you want to delete ${employeeName}?`)) {
            fetch(`<?= base_url('employees/delete/') ?>${employeeId}`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-Requested-With': 'XMLHttpRequest'
                }
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    window.location.reload();
                } else {
                    alert(data.message || 'Failed to delete employee');
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('An error occurred while deleting the employee');
            });
        }
    });
});

// Download all QR codes
document.getElementById('downloadAllQRBtn')?.addEventListener('click', async function() {
    const btn = this;
    const originalHTML = btn.innerHTML;

    // Show loading state
    btn.disabled = true;
    btn.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>Generating QR Codes...';

    try {
        // Fetch all employees with QR codes
        const response = await fetch('<?= base_url('employees/get-all-qr-codes') ?>', {
            method: 'GET',
            headers: {
                'X-Requested-With': 'XMLHttpRequest'
            }
        });

        if (!response.ok) {
            throw new Error('Failed to fetch QR codes');
        }

        const data = await response.json();

        if (!data.success || !data.employees || data.employees.length === 0) {
            alert(data.message || 'No employees with QR codes found');
            return;
        }

        // Generate PDF with all QR codes
        await generateQRCodesPDF(data.employees);

    } catch (error) {
        console.error('Error:', error);
        alert('An error occurred while generating QR codes');
    } finally {
        // Restore button state
        btn.disabled = false;
        btn.innerHTML = originalHTML;
    }
});

// Generate PDF with all QR codes
async function generateQRCodesPDF(employees) {
    const { jsPDF } = window.jspdf;
    const pdf = new jsPDF();

    const pageWidth = pdf.internal.pageSize.getWidth();
    const pageHeight = pdf.internal.pageSize.getHeight();
    const margin = 15;
    const qrSize = 50;
    const spacing = 10;
    const cardsPerRow = 3;
    const cardWidth = (pageWidth - (margin * 2) - (spacing * (cardsPerRow - 1))) / cardsPerRow;
    const cardHeight = 70;

    let x = margin;
    let y = margin;
    let employeeCount = 0;

    // Add title on first page
    pdf.setFontSize(16);
    pdf.setFont('helvetica', 'bold');
    pdf.text('Employee QR Codes', pageWidth / 2, y, { align: 'center' });
    y += 15;

    for (let i = 0; i < employees.length; i++) {
        const employee = employees[i];

        // Check if we need a new row
        if (employeeCount > 0 && employeeCount % cardsPerRow === 0) {
            x = margin;
            y += cardHeight + spacing;
        }

        // Check if we need a new page
        if (y + cardHeight > pageHeight - margin) {
            pdf.addPage();
            x = margin;
            y = margin;
        }

        // Draw card border
        pdf.setDrawColor(200, 200, 200);
        pdf.rect(x, y, cardWidth, cardHeight);

        // Add QR code image
        const qrUrl = `https://api.qrserver.com/v1/create-qr-code/?size=200x200&data=${encodeURIComponent(employee.qr_code)}`;

        try {
            // Fetch QR code image
            const imgResponse = await fetch(qrUrl);
            const imgBlob = await imgResponse.blob();
            const imgDataUrl = await new Promise((resolve) => {
                const reader = new FileReader();
                reader.onloadend = () => resolve(reader.result);
                reader.readAsDataURL(imgBlob);
            });

            // Add QR code to PDF
            const qrX = x + (cardWidth - qrSize) / 2;
            const qrY = y + 5;
            pdf.addImage(imgDataUrl, 'PNG', qrX, qrY, qrSize, qrSize);

            // Add employee name
            pdf.setFontSize(10);
            pdf.setFont('helvetica', 'bold');
            const nameY = qrY + qrSize + 5;
            pdf.text(employee.name, x + cardWidth / 2, nameY, { align: 'center', maxWidth: cardWidth - 4 });

            // Add employee number
            pdf.setFontSize(8);
            pdf.setFont('helvetica', 'normal');
            pdf.text(`#${employee.employee_number}`, x + cardWidth / 2, nameY + 4, { align: 'center' });

            // Add QR code text
            pdf.setFontSize(9);
            pdf.setFont('courier', 'bold');
            pdf.text(employee.qr_code, x + cardWidth / 2, nameY + 9, { align: 'center' });

        } catch (error) {
            console.error(`Error loading QR code for ${employee.name}:`, error);
        }

        x += cardWidth + spacing;
        employeeCount++;
    }

    // Save the PDF
    const fileName = `Employee_QR_Codes_${new Date().toISOString().split('T')[0]}.pdf`;
    pdf.save(fileName);
}
</script>

<!-- Load jsPDF library -->
<script src="https://cdnjs.cloudflare.com/ajax/libs/jspdf/2.5.1/jspdf.umd.min.js"></script>

<?= $this->endSection() ?>
